<?php

namespace App\Http\Services\Bill;

use App\Models\Product;
use Illuminate\Validation\Rule;
use App\Models\ProductWarehouse;
use App\Http\Enums\Bill\BillEnum;
use App\Http\Helper\CompanyHelper;
use App\Rules\ValidPaymentMethodRule;
use Illuminate\Validation\ValidationException;


class BillValidationService
{
    const ERR_PRICE = 'Price cannot be less than product base price.';
    const ERR_QUANTITY_LESS = 'Quantity cannot be less than product base quantity.';
    const ERR_QUANTITY_GREATER = 'Quantity cannot be greater than product base quantity.';
    const ERR_QUANTITY_NOT_ENOUGH = 'Quantity not enough in the warehouse.';


    public static function rules($type): array
    {
        return array_merge(
            static::commonRules(),
            static::getTypeRules($type),
        );
    }

    public static function getTypeRules($type): array
    {
        return $type == BillEnum::sales->value ? static::saleRules() : static::purchaseRules();
    }

    public static function commonRules()
    {
        $company_id = CompanyHelper::getCompany(request())->id;
        return [
            // main bill rules
            'bill_details.release_date' => 'required|date',
            'bill_details.total_bill_discount' => 'required|numeric|min:0',
            'bill_details.is_returned' => 'required|in:0,1',
            'bill_details.amount_paid' => 'required|numeric',
            'bill_details.branch_id' => 'required|exists:branches,id,company_id,' . $company_id,
            'bill_details.note' => 'required|string',
            'bill_details.customer_id' => 'required|integer',
            'bill_details.warehouse_id' => 'required|exists:warehouses,id,company_id,' . $company_id,
            'bill_details.warehouse_id' => [
                'required',
                'exists:warehouses,id,company_id,' . $company_id,
            ],

            // bill expenses
            'expenses' => 'nullable|array',
            'expenses.*.account_id' => 'required|distinct|exists:accounts,id,company_id,' . $company_id,
            'expenses.*.amount' => 'required|numeric|min:1',

            // bill payment methods
            'expenses_payment.payment_method' => [
                'required_if:expenses.*,>,0',
                'exists:payment_methods,id',
                new ValidPaymentMethodRule,
            ],

            // bill products
            'bill_products' => 'nullable|array',
            'bill_products.*.unit_id' => 'required|exists:company_units,id,company_id,' . $company_id,
            'bill_products.*.tax_included' => 'required|in:1,0',
            'bill_products.*.tax_id' => 'nullable|exists:taxes,id,company_id,' . $company_id,
            'bill_products.*.product_id' => 'required|exists:products,id,company_id,' . $company_id,
            'bill_products.*.price' => 'required|numeric|min:1',
            'bill_products.*.discount_type' => 'required|in:percentage,number',
            'bill_products.*.discount' => 'required|numeric|min:0',
            'bill_products.*.quantity' => 'required|numeric|min:0.001',
            // 'bill_products.*.price' => 'required|integer|min:1',

            // services
            'services' => 'nullable|array',
            'services.*.service_id' => 'required|exists:services,id',
            'services.*.discount_type' => 'required|in:number,presents',
            'services.*.discount_amount' => 'required|integer|min:0',
            'services.*.price' => 'nullable|integer|min:0',
            'services.*.date' => 'required|date',
            'services.*.quantity' => 'required|integer|min:1',
            'services.*.starts_at' => 'required|date_format:H:i:s',
            'services.*.tax_included' => 'required|in:1,0',
            'services.*.tax_id' => 'nullable|exists:taxes,id,company_id,' . $company_id,

            'bill_payment_methods' => 'array|required|min:1',
            // bill payment methods
            'bill_payment_methods.*.payment_method' => [
                'required',
                'exists:payment_methods,id',
                new ValidPaymentMethodRule,
            ],
            'bill_payment_methods.*.amount' => 'required|numeric',
            'bill_payment_methods.*.payment_method_type_id ' => 'nullable|exists:payment_method_types,id',
            'bill_payment_methods.*.phone' => 'nullable',

            // bill attachments
            'bill_details.attachment_file' => [
                'nullable',
                'file',
            ],
        ];
    }

    public static function saleRules()
    {
        return  [
            'bill_details.status' => 'required|string|in:draft,saved,accredited',
            'bill_details.date_of_supply' => 'required|date',
        ];
    }

    public static function purchaseRules()
    {
        return [];
    }

    public static function validateProductsQty($requested_products, $warehouse, $type, $is_invoice)
    {
        if (!$requested_products) {
            return;
        }

        foreach ($requested_products as $key => $requested_product) {

            $product = Product::where("is_stock","=",1)->where('id', $requested_product['product_id'])
                ->with(
                    [
                        'warehouses' => function ($query) use ($warehouse) {
                            $query->where('warehouse_id', $warehouse);
                        },
                        'units' => function ($query) use ($requested_product) {
                            $query->where('unit_id', $requested_product['unit_id']);
                        }
                    ]
                )
                ->first();
            

             if($product){
                    $product_unit = $product->units->first();

            //Create warehouse if the product does not have a warehouse
            $product_warehouse = $product->warehouses->first() ?? ProductWarehouse::create([
                'warehouse_id' => $warehouse,
                'product_id' => $requested_product['product_id'],
            ]);

            if (!$product_unit) {
                throw ValidationException::withMessages(['bill_products.' . $key . '.unit_id' => __('messages.product_unit_not_found')]);
            }

            $errors = [];
            if ($type == BillEnum::sales->value) {
                // validate min price
                if ($product_unit->selling_price > $requested_product['price']) {
                    $errors['bill_products.' . $key . '.price'] = __('messages.price_less_than_base_price');
                }

                // validate min qty
                if ($product_warehouse->min_quantity && $product_warehouse->min_quantity > $requested_product['quantity']) {

                    $errors['bill_products.' . $key . '.quantity'] = __('messages.quantity_less_than_base_quantity');
                }

                // validate qty in the warehouse
                if ($is_invoice && $product_warehouse->quantity  < $requested_product['quantity'] * $product_unit->conversion_factor) {

                    $errors['bill_products.' . $key . '.quantity'] = __(
                        'messages.quantity_not_enough_in_warehouse',
                        [
                            "model" =>
                            Product::find($requested_product['product_id'])->getTranslations('name')['ar']
                        ]
                    );
                }
            } else {
                // validate min qty
                if ($product_warehouse->max_quantity && $product_warehouse->max_quantity < $requested_product['quantity']) {
                    $errors['bill_products.' . $key . '.quantity'] = __('messages.quantity_greater_than_base_quantity');
                }
            }

            // Check if there are any errors
            if (!empty($errors)) {
                throw ValidationException::withMessages(array_unique($errors));
            }

                }else{
                    return 1;
                }
        }
    }
}
