<?php

namespace App\Http\Controllers\Api\Account;

use App\Models\Account;
use App\Trait\ApiTrait;
use Illuminate\Http\Request;
use App\Http\Helper\AccountHelper;
use App\Http\Helper\CompanyHelper;
use App\Http\Controllers\Controller;
use App\Http\Resources\AccountResource;
use App\Http\Services\Account\AccountService;
use App\Http\Requests\Api\Account\StoreAccountRequest;
use App\Http\Requests\Api\Account\SearchAccountRequest;
use App\Http\Requests\Api\Account\UpdateAccountRequest;
use Illuminate\Validation\ValidationException;


class AccountController extends Controller
{
    use ApiTrait;

    public function index(Request $request)
    {
        $this->authorize('view accounts');

        return $this->apiResponse(
            data: AccountResource::collection(
                Account::whereBelongsTo(CompanyHelper::getCompany())
                    ->with([
                        'children',
                        'models',
                        'entries' => function ($query) use ($request) {
                            return $query
                                ->when(
                                    $request->has('from'),
                                    fn($q) => $q->whereDate('created_at', '>=', $request->input('from'))
                                )
                                ->when(
                                    $request->has('to'),
                                    fn($q) => $q->whereDate('created_at', '<=', $request->input('to'))
                                );
                        },
                    ])
                    ->whereNull('parent_id')
                    ->get()
            )
        );
    }

    public function getAllAccounts()
    {
        $this->authorize('view accounts');
        return $this->apiResponse(data: AccountResource::collection(
            Account::whereBelongsTo(CompanyHelper::getCompany())->get()
        ));
    }
    public function show($id)
    {

        $this->authorize('show accounts');
        $account = Account::withTrashed()->findOrFail($id);
        $account->load([
            'children',
            'entries.journal',
            'entries.type',
        ]);

        return $this->apiResponse(data: AccountResource::make($account));
    }
    public function getExpensesAccounts()
    {
        $account = Account::where([
            ['code', 5],
            ['company_id', CompanyHelper::getId()],
        ])->first();
        $account->load([
            'children',
            'entries.journal',
            'entries.type',
        ]);


        return $this->apiResponse(data: AccountResource::make($account));
    }

    public function store(StoreAccountRequest $request, Account $account)
    {
        $this->authorize('create accounts');
        return $this->apiResponse(
            message: __('messages.account_created_successfully'),
            data: AccountResource::make(
                CompanyHelper::getCompany(request())->accounts()->create([
                    'code' => AccountService::generateCode($account),
                    'level_id' => AccountService::$nextLevel->id
                ] + $request->validated())
            )
        );
    }

    public function update(UpdateAccountRequest $request, Account $account)
    {
        $this->authorize('edit accounts');
        if ($request->parent_id !== null) {
            AccountService::transferAccount($account, Account::find($request->parent_id));
        }
        if ($account->status == 'sub' && $request->validated()['status'] == 'main' && $account->entries()->first()) {
            throw ValidationException::withMessages(['account' => [__('messages.cannot_delete_account_with_entries')]]);
        }
        $account->update($request->validated());
        return $this->apiResponse(message: __('messages.account_updated_successfully'), data: AccountResource::make($account));
    }

    // public function destroy($id)
    // {
    //     $this->authorize('delete accounts');
    //     $account = Account::withTrashed()->findOrFail($id);
    //     AccountService::deleteAccount($account);
    //     return $this->apiResponse(message: __('messages.account_deleted_successfully'));
    // }
    public function destroy(Request $request)
    {
       $this->authorize('delete accounts');

        // Validate the input
        $validated = $request->validate([
            'account_ids' => 'required|array|min:1',
            'account_ids.*' => 'exists:accounts,id',
            'force' => 'required|boolean',
        ]);

        $employeeIds = $validated['account_ids'];
        $forceDelete = $validated['force'];

        foreach ($employeeIds as $empId) {
            $account = Account::withTrashed()->findOrFail($empId);

            // Use the deleteAccount method for each account
            AccountService::deleteAccount($account, $forceDelete);
        }

        $message = $forceDelete
            ? __('messages.employees_force_deleted_successfully')
            : __('messages.employees_deleted_successfully');

        return $this->apiResponse(message: $message);
    }

    public function search(SearchAccountRequest $request)
    {
        $this->authorize('view accounts');
        $accounts = Account::where('company_id', auth()->id())
            ->when($request->has('code'), fn($q) => $q->codeSearch($request->code))
            ->when($request->has('name'), fn($q) => $q->nameSearch($request->name))
            ->get();

        return $this->apiResponse(data: AccountResource::collection($accounts));
    }

    public function createPreStructuredAccounts()
    {
        AccountHelper::copyAccounts();
        return $this->apiResponse(message: __('messages.account_clone_successfully'));
    }

    public function transferAccount(Account $account, Account $parent)
    {
        AccountService::transferAccount($account, $parent);
        return $this->apiResponse(message: __('messages.account_transfer_successfully'));
    }

    public function toggleArchiveAccount($id)
    {
        $account = Account::withTrashed()->findOrFail($id);

        if ($account->trashed()) {
            $account->restore();
            return $this->apiResponse(message: __('messages.account_restored_successfully'));
        }
        $account->delete();
        return $this->apiResponse(message: __('messages.account_archive_successfully'));
    }
    public function getArchivedAccounts()
    {
        return $this->apiResponse(data: AccountResource::collection(
            Account::whereBelongsTo(CompanyHelper::getCompany())->Archived()->get()
        ));
    }
}
