<?php

namespace App\Http\Controllers\Api\Auth;

use Carbon\Carbon;
use App\Models\Company;
use App\Models\Package;
use App\Trait\ApiTrait;
use App\Events\NewCompany;
use App\Trait\UploadFileTrait;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use App\Http\Enums\Subscription\PaymentStatusEnum;
use App\Http\Helper\CompanyHelper;
use App\Http\Requests\Api\Company\RegisterCompanyRequest;
use App\Http\Requests\Api\Company\VerifyEmailRequest;
use App\Jobs\SendOtpEmailVerification;
use App\Models\UserOtp;
use Illuminate\Validation\ValidationException;

class EmailVerificationController extends Controller
{
    use ApiTrait;

    public function sendOtp()
    {
        self::sendNewOtp();

        return $this->apiResponse(message: __('messages.verification_code_sent_successfully'));
    }

    public function verify(VerifyEmailRequest $request)
    {
        $company = CompanyHelper::getCompany();

        if ($company->verified()) {
            throw ValidationException::withMessages([
                'email' => __('messages.email_already_verified'),
            ]);
        }

        $otp = UserOtp::where('email', $company->email)->first();

        if (!$otp || $otp->code != $request->code) {
            throw ValidationException::withMessages([
                'otp' => __('messages.invalid_code'),
            ]);
        }

        if (now() > $otp->expire_at) {
            self::sendNewOtp();
            throw ValidationException::withMessages([
                'otp' => __('messages.code_expired_and_email_send'),
            ]);
        }

        $company->update(['email_verified_at' => now()]);

        $otp->delete();

        return $this->apiResponse(message: __('messages.email_verified_successfully'));
    }

    private static function sendNewOtp()
    {
        $company = CompanyHelper::getCompany();

        if ($company->verified()) {
            throw ValidationException::withMessages([
                'email' => __('messages.email_already_verified'),
            ]);
        }

        UserOtp::where('email', $company->email)->delete();

        SendOtpEmailVerification::dispatch($company->email);
    }
}
