<?php

namespace App\Http\Controllers\Api\Bound;

use App\Trait\ApiTrait;
use App\Http\Controllers\Controller;
use App\Http\Enums\Bound\BoundTypeEnum;
use App\Http\Helper\CompanyHelper;
use App\Http\Resources\BoundResource;
use App\Http\Services\Bound\BoundService;
use App\Http\Requests\Api\Bound\BoundRequest;
use Illuminate\Http\Request;
use App\Models\Bound;

class BoundController extends Controller
{
    use ApiTrait;

    public function index()
    {
        $this->authorize('view bounds');

        return $this->apiResponse(
            data: BoundResource::collection(
                CompanyHelper::getCompany()->bounds()
                    ->where('type', BoundTypeEnum::getValueFromString(request()->route()->type))
                    ->when(request()->filled('voucher_type'), function ($query) {
                        $query->where('voucher_type', request()->get('voucher_type'));
                    })                    
                    ->with(['entry:id,name', 'payment:id,name'])
                    ->get()
            ),
        );
    }

    public function show(string $type, Bound $bound)
    {
        $this->authorize('show bounds');

        $bound->load(['entry', 'payment:id', 'paymentMethod:id','bills']);

        return $this->apiResponse(
            data: BoundResource::make($bound),
        );
    }

    public function store(BoundRequest $request, $type)
    {
        $this->authorize('create bounds');
        $boundService=new BoundService;
        
        $boundService->create($request, $type);
        // $boundService->updateBill($request);

        return $this->apiResponse(
            message: __('messages.bound_created_successfully'),
        );
    }

    public function update(BoundRequest $request, $type, Bound $bound)
    {
        $this->authorize('create bounds');

        (new BoundService)->update($request, $bound, $type);

        return $this->apiResponse(
            message: __('messages.bound_updated_successfully'),
        );
    }

    public function destroy(string $type, Bound $bound)
    {
        $this->authorize('delete bounds');
        (new BoundService)->delete($bound);
        return $this->apiResponse(
            message: __('messages.bound_deleted_successfully'),
        );
    }
    
    // public function toggleArchive($type, $id)
    // {
    //     $bound = Bound::withTrashed()->findOrFail($id);
    
    //     if ($bound->trashed()) {
    //         $bound->restore();
    //         return response()->json(['message' => __('messages.bound_restored_successfully')]);
    //     }
    
    //     $bound->delete();
    //     return response()->json(['message' => __('messages.bound_archived_successfully')]);
    // }
    public function toggleArchive($type, Request $request)
    {
        $this->validate($request, [
            'bound_ids' => 'required|array',
            'bound_ids.*' => 'integer|exists:bounds,id',
        ]);
    
        $service = app(BoundService::class);
    
        foreach ($request->bound_ids as $id) {
            $bound = Bound::withTrashed()->findOrFail($id);
            $service->toggleArchive($bound);
        }
    
        return response()->json([
            'message' => __('messages.operation_completed_successfully'),
        ]);
    }
    
    // public function archived($type)
    // {
    //     $bounds = Bound::onlyTrashed()
    //         ->where('type', $type)
    //         ->where('company_id', auth('company')->id()) // adjust for auth
    //         ->with('boundEntries') // eager load if needed
    //         ->get();
    
    //     return response()->json($bounds);
    // }
    public function archived($type)
    {
        $this->authorize('view bounds');
    
        $typeValue = BoundTypeEnum::getValueFromString($type);

        $bounds = Bound::onlyTrashed()
            ->where('type', $typeValue)
            ->where('company_id', CompanyHelper::getCompany()->id)
            ->when(request()->filled('voucher_type'), function ($query) {
                $query->where('voucher_type', request()->get('voucher_type'));
            })            
            ->with(['boundEntries'])
            ->get();
    
        return $this->apiResponse(data: BoundResource::collection($bounds));
    }

    //   protected static function getLastBoundNum()
    // {
    //     $code = self::generateRandomNumber();

    //     $isUnique = false;

    //     while (!$isUnique) {
    //         $existingUser = Bound::where('bound_number', $code)->exists();

    //         if (!$existingUser) {
    //             $isUnique = true;
    //         } else {
    //             $code = self::generateRandomNumber();
    //         }
    //     }

    //     return response()->json([
    //     'status' => true,
    //     'data' => $code,
    // ]);
    // }
    //  private static function generateRandomNumber(): string
    // {
    //     return (string) mt_rand(0, 99999999);
    // }
}
