<?php

namespace App\Http\Controllers\Api\Company;

use App\Events\NewCompany;
use App\Http\Requests\UpdateCompanyPasswordRequest;
use App\Models\Company;
use App\Models\Subscription;
use App\Http\Controllers\Controller;
use App\Http\Requests\Api\Company\StoreCompanyRequest;
use App\Http\Requests\Api\Company\UpdateCompanyRequest;
use App\Http\Requests\SendCompanyMailRequest;
use App\Http\Requests\UpdateCompanyPackageRequest;
use App\Http\Resources\CompanyPackageResource;
use App\Mail\AdminCompanyEmail;
use App\Models\Package;
use App\Trait\ApiTrait;
use App\Trait\UploadFileTrait;
use Carbon\Carbon;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;

class CompanyController extends Controller
{

    use ApiTrait, UploadFileTrait;

    public function index()
    {
        // $this->authorize('viewAny',  auth()->user());
        $companies = Company::get();
        return $this->apiResponse(data: CompanyPackageResource::collection($companies));
    }


    public function store(StoreCompanyRequest $request)
    {

        $logo = $this->uploadFile(Company::UPLOADED_FILES, $request->logo);

        $company = Company::create([
            'logo' => $logo,
            'password' => Hash::make($request->password)
        ] + $request->validated());

        event(new NewCompany($company));
        return $this->apiResponse(data: $company, message: __('messages.company_created_successfully'));
    }

    public function show(Company $company)
    {

        // $this->authorize('view', ['company' => $company]);
        return $this->apiResponse(data: $company);
    }


    public function update(UpdateCompanyRequest $request, Company $company)
    {
        
            $validatedData = $request->validated();
            if ($request->has('password')) {
                $validatedData['password'] = Hash::make($request->validated(['password']));
            }
        // $this->authorize('edit company', auth()->user());
        $company->update(    $validatedData );
        return $this->apiResponse(data: $company, message: __('messages.company_updated_successfully'));
    }

    public function destroy(Company $company)
    {
        // $this->authorize('delete company', auth()->user());
        $company->delete();
        return $this->apiResponse(message: __('messages.company_deleted_successfully'));
    }

    public function updateCompanyPackage(UpdateCompanyPackageRequest $request)
    {

        $companySubscription = Subscription::where('company_id', $request->company_id)->first();

        $package = Package::find($request->package_id);

        $companySubscription->package_id = $package->id;
        $companySubscription->start_date = Carbon::now();
        $companySubscription->end_date = Carbon::now()->addDays($package->period);
        $companySubscription->price = $package->price;
        $companySubscription->save();
        return $this->apiResponse(data: [$companySubscription], message: __('messages.company_package_updated_successfully'));
    }

    public function sendMail(SendCompanyMailRequest $request)
    {
        $company_email = Company::find($request->company_id)->email;
        Mail::to($company_email)->send(new AdminCompanyEmail($request->title, $request->body));
        return $this->apiResponse(message: __('messages.admin_company_email_sent_successfully'));
    }
    public function updatePassword(UpdateCompanyPasswordRequest $request, Company $company)
    {
        $company->update([
            'password' => Hash::make($request->password)
        ]);

        return $this->apiResponse(message: __('messages.company_password_updated_successfully'));
    }
}
