<?php

namespace App\Http\Controllers\Api\Company\Journal;

use App\Models\Branch;
use App\Models\Client;
use App\Models\Journal;
use App\Models\Product;
use App\Models\Supplier;
use App\Models\Warehouse;
use App\Models\JournalEntry;
use Illuminate\Http\Request;
use App\Http\Helper\CompanyHelper;
use App\Http\Controllers\Controller;
use App\Http\Resources\EntryResource;
use App\Http\Resources\JournalResource;
use App\Http\Services\Account\JournalService;
use Illuminate\Validation\ValidationException;
use App\Http\Requests\Api\Journal\StoreJournalRequest;
use App\Http\Requests\Api\Journal\JournalReportRequest;
use App\Http\Requests\Api\Journal\UpdateJournalRequest;
use App\Models\JournalFile;
use App\Trait\UploadFileTrait;

class JournalController extends Controller
{
    use UploadFileTrait;

    public function index(Request $request)
    {
        $this->authorize('view journals');

        return $this->apiResponse(
            data: JournalResource::collection(
                CompanyHelper::getCompany(request())
                    ->journals()
                    ->doesntHave('journalRepeation')
                    ->when(
                        $request->has('report'),
                        fn($q) => $q->with([
                            'entries.account' => function ($query) use ($request) {
                                if ($request->has('account')) {
                                    $accountAndItsChildrenIds = CompanyHelper::getCompany()
                                        ->accounts()
                                        ->where('parent_id', $request->account)
                                        ->pluck('id')
                                        ->toArray();

                                    $accountAndItsChildrenIds[] = (int) $request->account;

                                    $query->whereIn('id', $accountAndItsChildrenIds);
                                }
                            },
                            'entries.type',
                            'journalable',
                            'employee',
                        ])
                    )
                    ->when(
                        $request->has('from'),
                        fn($q) => $q->whereDate('created_at', '>=', $request->input('from'))
                    )
                    ->when(
                        $request->has('to'),
                        fn($q) => $q->whereDate('created_at', '<=', $request->input('to'))
                    )->with('entries.account')
                     ->orderBy('date', 'asc') 
                    ->get()
            ),
        );
    }

    public function store(StoreJournalRequest $request)
    {

        $this->authorize('create journals');

        $debit_side = JournalService::handleSide(
            JournalService::filterAccountsSide(
                accounts: $request->accounts,
                side: 'debit'
            ),
            'debit'
        );

        $credit_side = JournalService::handleSide(
            JournalService::filterAccountsSide(
                accounts: $request->accounts,
                side: 'credit'
            ),
            'credit'
        );

        if ($debit_side['side_amount'] !== $credit_side['side_amount']) {
            throw ValidationException::withMessages(['debit' => [__('messages.journal_validation_error')]]);
        }



        $file = $this->uploadFile(Journal::UPLOADED_FILES, $request->file);

        $journal = JournalService::createJournal(
            date: $request->date ?? now(),
            type: __('constants.journal_manual_type'),
            source: "Manual",
            description: $request->description,
            file: $file,
            employee: null,
            status: $request->status,
            debit: $debit_side['side_data'],
            credit: $credit_side['side_data'],
            reference_code: $request->reference_code,
            branch: Branch::findOr($request->branch_id, fn() => null),
            warehouse: Warehouse::findOr($request->warehouse_id, fn() => null),
            repeatable: $request->repeatable,
            repeatableData: $request->repeatable ? $this->getRepeatableJournalDataFromRequest($request) : null,
        );
        
        if ($request->has('files')) {
        foreach ($request->file('files') as $fileData) {
                $uploadedFile = $this->uploadFile(Journal::UPLOADED_FILES, $fileData['file']);

                $journal->files()->create([
                    'file' => $uploadedFile,
                    'reference_number' => $fileData['reference_number'] ?? null,
                ]);
            
        }
    }

        $journal->load(['entries.account']);

        return $this->apiResponse(
            data: new JournalResource($journal),
            message: __('messages.journal_created_successfully'),
        );
    }

    public function show(Journal $journal)
    {
        $this->authorize('show journals');

        $journal->load([
            'entries.account',
            'entries.type',
            'journalable',
            'employee',
            'journalRepeation',
            'files',
        ]);

        return $this->apiResponse(
            data: new JournalResource($journal),
        );
    }
    public function update(UpdateJournalRequest $request, Journal $journal)
    {
        if ($journal->status == 'accredited') {

            throw   ValidationException::withMessages([
                'message' => __('messages.journal_accredited_cannot_update')
            ]);
        }

        (new JournalService())->updateJournal(
            $request,
            Journal::where(
                'company_id',
                CompanyHelper::getId()
            )->where('id', $journal->id)->firstOrFail(),
            journalRepeation: null,
            repeatable: $request->repeatable,
            status: $request->status,
            repeatableData: $request->repeatable ? $this->getRepeatableJournalDataFromRequest($request) : null
        );
        return $this->apiResponse(
            message: __('messages.journal_updated_successfully'),
        );
    }

    public function destroy(Journal $journal)
    {

        JournalService::deleteJournal(Journal::where('company_id', CompanyHelper::getId())->where('id', $journal->id)->firstOrFail());
        return $this->apiResponse(message: __('messages.journal_deleted_successfully'));
    }
    public function reports(JournalReportRequest $request)
    {
        $this->authorize('view journals');

        $entries =
            JournalEntry::whereHas('journal', function ($q) use ($request) {
                $q->where('company_id', CompanyHelper::getId());

                if ($request->has('branch') && !empty($request->branch)) {
                    $q->whereIn('branch_id', $request->branch);
                }

                if ($request->has('warehouse') && !empty($request->warehouse)) {
                    $q->whereIn('warehouse_id', $request->warehouse);
                }
            })
            ->when($request->has('from'), fn($q) => $q->whereDate('created_at', '>=', $request->input('from')))
            ->when($request->has('to'), fn($q) => $q->whereDate('created_at', '<=', $request->input('to')))
            ->when(
                $request->has('product') && !empty($request->product),
                function ($query) use ($request) {
                    $query->whereHasMorph(
                        'type',
                        [Product::class],
                        fn($q) => $q->whereIn('id', $request->product)
                    );
                }
            )
            ->when(
                $request->has('client') && !empty($request->client),
                function ($query) use ($request) {
                    $query->whereHasMorph(
                        'type',
                        [Client::class],
                        fn($q) => $q->whereIn('id', $request->client)
                    );
                }
            )
            ->when(
                $request->has('supplier') && !empty($request->supplier),
                function ($query) use ($request) {
                    $query->whereHasMorph(
                        'type',
                        [Supplier::class],
                        fn($q) => $q->whereIn('id', $request->supplier)
                    );
                }
            )
            ->when(
                $request->has('account'),
                function ($query) use ($request) {
                    $query->whereHas(
                        'account',
                        function ($query) use ($request) {

                            $accountChildrenIds = CompanyHelper::getCompany()
                                ->accounts()
                                ->where('parent_id', (int) $request->account)
                                ->pluck('id')
                                ->toArray();

                            $query->whereIn('id', $accountChildrenIds);
                        }
                    );
                }
            )
            ->with(
                [
                    'type',
                    'account',
                    'journal',
                    'journal.branch',
                    'journal.warehouse',
                ]
            )
            ->get();

        return $this->apiResponse(
            data: EntryResource::collection($entries),
        );
    }

    public function reverseJournal(Journal $journal)
    {
        JournalService::reverseJournal($journal);
        return $this->apiResponse(message: __('messages.journal_reversed_successfully'));
    }


    private function getRepeatableJournalDataFromRequest($request)
    {
        return [
            'period' => $request->input('period'),
            'start_date' => $request->input('start_date'),
            'end_date' => $request->input('end_date'),
            'desc' => $request->input('desc'),
        ];
    }
    //////////////////
     public function getFiles(Journal $journal)
    {
        $files = $journal->files()->get();

        return $this->apiResponse(
            data: $files,
            message: __('messages.files_retrieved_successfully')
        );
    }

    public function uploadJournalFile(Request $request, Journal $journal)
    {
        $request->validate([
            'file'             => 'required|file',
            // 'reference_number' => 'nullable|string',
            'reference_number' => 'nullable|string|unique:journal_files,reference_number',

        ]);

        $fileName = $this->uploadFile(Journal::UPLOADED_FILES, $request->file('file'));

        if (!$fileName) {
            return $this->apiResponse(
                message: __('messages.file_upload_failed')
            );
        }

        $journalFile = $journal->files()->create([
            'file'             => $fileName,
            'reference_number' => $request->input('reference_number'),
        ]);

        return $this->apiResponse(
            data: $journalFile,
            message: __('messages.file_uploaded_successfully')
        );
    }

    public function deleteJournalFile(Journal $journal, JournalFile $file)
    {
        if ($file->journal_id !== $journal->id) {
            return $this->apiResponse(
                message: __('messages.file_not_belong_to_journal')
            );
        }

        $this->removeFile(Journal::UPLOADED_FILES, $file->file);

        $file->delete();

        return $this->apiResponse(
            message: __('messages.file_deleted_successfully')
        );
    }
}
