<?php

namespace App\Http\Controllers\Api\Company;

use App\Models\Bill;
use App\Models\Shift;
use App\Trait\ApiTrait;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Resources\BillResource;
use App\Http\Resources\ShiftResource;
use App\Http\Requests\Api\Shift\StoreShiftRequest;
use App\Http\Requests\Api\Shift\UpdateShiftRequest;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use App\Http\Services\Worker as WorkerService;

class ShiftController extends Controller
{
    use ApiTrait;

    public function index(Request $request)
    {
        $this->authorize('view shifts');

        return $this->apiResponse(
            data: ShiftResource::collection(
                // Shift::when($request->has('branch'), fn ($q) => $q->where('branch_id', $request->branch))
                //     ->when($request->has('treasury'), fn ($q) => $q->where('treasury_id', $request->treasury))
                //     ->when($request->has('employee'), fn ($q) => $q->where('employee_id', $request->employee))
                Shift::whereHas('employee', function ($query) {
                    $query->where('company_id',   auth('company')->id() ?? auth('worker')->user()->company->id);
                })->with(
                    [
                        'branch:id,name',
                        'treasury:id,name',
                        'employee:id,name,code',
                    ]
                )->get()
            )
        );
    }

    public function show(Shift $shift)
    {
        $this->authorize('show shifts');

        $bills = Bill::with(
            [
                'billPaymentMethods',
                'billProducts.product',
                'billProducts.tax',
                'billServices.service',
                'billServices.tax',
            ]
        )
            ->whereHas(
                'journals',
                fn ($q) => $q->where('shift_id', $shift->id)
            )
            ->with(
                [
                    'billable',
                    'journals',
                ]
            )
            ->get();

        $journalsIds = [];

        $bills->each(function ($bill) use (&$journalsIds) {
            $journalsIds = array_merge($journalsIds, $bill->journals->pluck('id')->toArray());
        });

        $shift->load(
            [
                'journals' => fn ($q) => $q->whereNotIn('id', $journalsIds),
                'journals.entries.type',
                'journals.entries.account',
                'branch:id,name',
                'treasury:id,name',
                'autoShift',
                'employee:id,name,code',
            ]
        );

        return $this->apiResponse(data: [
            'shift' => ShiftResource::make($shift),
            'bills' => BillResource::collection($bills),
        ]);
    }

    public function store(StoreShiftRequest $request)
    {
        $this->authorize('create shifts');

        WorkerService\ShiftPermissionAfterLogin::removePermissionFromWorker();


        Shift::create(
            [
                'open_name' => $request->user()->name ?? $request->user()->employee->name,
            ] + $request->validated()
        );

        return $this->apiResponse(message: __('messages.shift_created_successfully'));
    }

    public function update(UpdateShiftRequest $request, Shift $shift)
    {
        $this->authorize('edit shifts');

        $shift->update(
            [
                'end_amount' => $request->end_amount,
                'close_name' => $request->user()->name ?? $request->user()->employee->name,
            ]
        );

        return $this->apiResponse(message: __('messages.shift_updated_successfully'));
    }
}
