<?php

namespace App\Http\Controllers\Api\Company\Worker;

use App\Models\Worker;
use App\Trait\ApiTrait;
use Illuminate\Http\Request;
use App\Trait\UploadFileTrait;
use Spatie\Permission\Models\Role;
use App\Http\Controllers\Controller;
use App\Http\Helper\CompanyHelper;
use App\Http\Resources\WorkerResource;
use App\Http\Requests\Api\Company\Worker\StoreWorkerRequest;
use App\Http\Requests\Api\Company\Worker\UpdateWorkerRequest;

class WorkerController extends Controller
{
    use ApiTrait, UploadFileTrait;

    public function index(Request $request)
    {
        $this->authorize('view workers');

        return $this->apiResponse(data: WorkerResource::collection(CompanyHelper::getCompany()->workers()->with(['employee', 'roles.permissions'])->get()));
    }

    public function store(StoreWorkerRequest $request)
    {
        $this->authorize('create workers');

        if ($request->has('image')) {
            $image = $this->uploadFile(Worker::UPLOADED_FILES, $request->image);
        }
        $worker = CompanyHelper::getCompany()->workers()->create(['image' => $image ?? null] + $request->validated());
        $worker->assignRole(Role::find($request->role_id));

        return $this->apiResponse(data: WorkerResource::make($worker), message: __('messages.worker_created_successfully'));
    }

    public function show(Worker $worker)
    {
        $this->authorize('show workers');

        $worker->load([
            'employee',
            'roles.permissions',
        ]);

        return $this->apiResponse(data: new WorkerResource($worker));
    }

    public function update(UpdateWorkerRequest $request, Worker $worker)
    {
        $this->authorize('edit workers');

        $image = $request->has('image') ? $this->uploadFile(Worker::UPLOADED_FILES, $request->image, $worker->image) : $worker->image;

        $worker->update(['image' => $image] + $request->validated());
        //remove the old role first
        $worker->roles()->detach();
        // assign the new role
        $worker->assignRole(Role::find($request->role_id));
        return $this->apiResponse(data: new WorkerResource($worker), message: __('messages.worker_updated_successfully'));
    }

    public function destroy(Worker $worker)
    {
        $this->authorize('delete workers');
        $worker->delete();
        return $this->apiResponse(message: __('messages.worker_deleted_successfully'));
    }
}
