<?php

namespace App\Http\Controllers\Api\Product;

use App\Models\Product;
use App\Trait\ApiTrait;
use App\Http\Helper\CompanyHelper;
use App\Http\Controllers\Controller;
use App\Http\Services\ProductService;
use App\Http\Resources\ProductResource;
use App\Http\Requests\Api\Product\StoreProductRequest;
use App\Http\Requests\Api\Product\UpdateProductRequest;
use App\Http\Requests\DeleteProductRequest;

class ProductController extends Controller
{
    use ApiTrait;

    public function __construct(public ProductService $productService) {}

    public function index()
    {
        $this->authorize('view products');

        // Get the status from the request, default to null if not present
        $status = request()->query('status');

        // Build the query
        $query = Product::whereBelongsTo(CompanyHelper::getCompany(request()))
            ->with([
                'units',
                'tax',
                'warehouses.warehouse',
                'warehouses.warehouseUnits.unit',
                'branchs.branchUnits.unit',
                'category',
            ]);

        // Add the status condition if provided
        if (!is_null($status)) {
            $query->where('status', $status);
        }

        // Execute the query and return the response
        $products = $query->latest()->get();

        return $this->apiResponse(
            data: ProductResource::collection($products)
        );
    }

    public function show(Product $product)
    {
        $this->authorize('show products');

        $product->load([
            'tax',
            'warehouses.warehouse',
            'warehouses.warehouseUnits.unit',
            'branchs.branch',
            'branchs.branchUnits.unit',
            'units',
        ]);

        return $this->apiResponse(data: ProductResource::make($product));
    }

    public function store(StoreProductRequest $request)
    {
        $this->authorize('create products');

        return $this->apiResponse(
            message: __('messages.product_created_successfully'),
            data: ProductResource::make($this->productService->create($request))
        );
    }

    public function update(UpdateProductRequest $request, Product $product)
    {
        $this->authorize('edit products');

        $product = $this->productService->update($request, $product);
        return $this->apiResponse(message: __('messages.product_updated_successfully'), data: ProductResource::make($product));
    }

    public function changeStatus(Product $product)
    {
        $this->authorize('edit products');

        $product->update([
            'status' => !$product->status
        ]);
        return $this->apiResponse(message: __('messages.product_status_changed_successfully'), data: ProductResource::make($product));
    }

    public function generateUniqueKey()
    {
        $this->authorize('create products');

        return $this->apiResponse(
            data: [
                'key' => ProductService::generateUniqueKey()
            ]
        );
    }

    public function destroy(DeleteProductRequest $request, Product $product = null)
    {
        $this->authorize('delete products');
        if ($product) {
            $product->forceDelete();
            return $this->apiResponse(message: __('messages.product_deleted_successfully'));
        } else {
            if (is_array($request->ids)) {
                CompanyHelper::getCompany()->products()->whereIn('id', $request->ids)->delete();
                return $this->apiResponse(message: __('messages.products_deleted_successfully'));
            }
        }
    }
    public function deleteAll()
    {
        $this->authorize('delete products');
        Product::where('company_id', CompanyHelper::getId())->forceDelete();
        return $this->apiResponse(message: __('messages.products_deleted_successfully'));
    }
}
