<?php

namespace App\Http\Controllers\Api\Shipments;

use App\Http\Controllers\Controller;
use App\Http\Helper\CompanyHelper;
use App\Http\Requests\AssignShipmentToDeliveryRequest;
use App\Http\Requests\ShipmentRequest;
use App\Http\Requests\UpdateShipmentStatusRequest;
use App\Http\Resources\ShipmentResource;
use App\Http\Services\NotificationService;
use App\Http\Services\Shipments\ShipmentService;
use App\Models\Shipment;
use App\Trait\ApiTrait;
use Illuminate\Database\Eloquent\ModelNotFoundException;

class ShipmentController extends Controller
{
    use ApiTrait;

    public function index()
    {
        $this->authorize('view shipments');
        return $this->apiResponse(data: ShipmentResource::collection(Shipment::where('company_id', CompanyHelper::getId())->get()));
    }

    public function show(Shipment $shipment)
    {
        $this->authorize('show shipments');
        $shipment = Shipment::where('id', $shipment->id)
            ->where('company_id',  CompanyHelper::getId())
            ->first();
        return $this->apiResponse(data: ShipmentResource::make($shipment));
    }


    public function store(ShipmentRequest $request)
    {
        $this->authorize('create shipments');
        return $this->apiResponse(data: ShipmentResource::make(ShipmentService::create($request->validated())), message: __('messages.shipment_created_successfully'));
    }

    public function update(ShipmentRequest $request, Shipment $shipment)
    {
        $this->authorize('edit shipments');
        $shipment->update($request->validated());
        return $this->apiResponse(data: ShipmentResource::make($shipment), message: __('messages.shipment_updated_successfully'));
    }

    public function destroy(Shipment $shipment)
    {
        $this->authorize('delete shipments');
        $shipment = Shipment::where('id', $shipment->id)
            ->where('company_id',  CompanyHelper::getId())
            ->first();
        $shipment->delete();
        return $this->apiResponse(message: __('messages.shipment_deleted_successfully'));
    }
    public function updateShipmentStatus(UpdateShipmentStatusRequest $request, $id)
    {
        $this->authorize('updateShipmentStatus shipments');
        $shipment = Shipment::where('id', $id)
            ->where('company_id',  CompanyHelper::getId())
            ->first();
        $shipment->update([
            'status' => $request->status,
        ]);
        return $this->apiResponse(data: ShipmentResource::make($shipment), message: __('messages.shipment_status_updated_successfully'));
    }

    public function acceptShipmentFromClient($id)
    {
        $this->authorize('accept shipments');
        $shipment = Shipment::where('id', $id)
            ->where('company_id',  CompanyHelper::getId())
            ->where('status',  1)
            ->first();
        if (!$shipment) {
            throw  new ModelNotFoundException();
        }
        $shipment->update([
            'status' => 2,
        ]);
        return $this->apiResponse(message: __('messages.shipment_accepted'));
    }
    public function assignShipmentForDelivery(AssignShipmentToDeliveryRequest $request, $id)
    {
        $this->authorize('assign shipments');

        $shipment = Shipment::where('id', $id)
            ->where('company_id',  CompanyHelper::getId())
            ->where('status',  2)
            ->first();

        if (!$shipment) {
            throw  new ModelNotFoundException();
        }
        $shipment->status = 3;
        $shipment->delivery_id = $request->delivery_id;
        $shipment->save();

        NotificationService::sendNotificationToDelivery($request->delivery_id);
        return $this->apiResponse(message: __('messages.shipment_assigned'));
    }
}
