<?php

namespace App\Http\Controllers\Api\Supplier;

use App\Models\Address;
use App\Trait\ApiTrait;
use App\Models\Supplier;
use App\Http\Helper\CompanyHelper;
use App\Http\Controllers\Controller;
use App\Http\Resources\SupplierResource;
use App\Http\Requests\Api\Supplier\StoreSupplierRequest;
use App\Http\Requests\Api\Supplier\UpdateSupplierRequest;
use App\Http\Requests\Api\Supplier\ChangeSupplierStatusRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class SupplierController extends Controller
{
    use ApiTrait;

    public function index(Request $request)
    {
        $this->authorize('view suppliers');

        return $this->apiResponse(
            data: SupplierResource::collection(
                Supplier::whereBelongsTo(CompanyHelper::getCompany(request()))
                    ->whereDoesntHave('blocked')
                    ->when(request()->has('active') && request()->active == true, fn($q) => $q->active())
                    ->when($request->has('special'), fn($q) => $q->where('special', true), fn($q) => $q->where('special', false))
                    ->with(
                        [
                            'address.area.city.country',
                            'entries',
                            'bills.billPaymentMethods'
                        ]
                    )->get()
            )
        );
    }

    // public function store(StoreSupplierRequest $request)
    // {
    //     $this->authorize('create suppliers');

    //     $address = Address::create($request->only(['post_code', 'address_line', 'area_id']));
    //     return $this->apiResponse(
    //         message: __('messages.supplier_created_successfully'),
    //         data: SupplierResource::make(
    //             Supplier::create(
    //                 ['address_id' => $address->id] + $request->validated()
    //             )
    //         )
    //     );
    // }
    public function store(StoreSupplierRequest $request)
    {
        $this->authorize('create suppliers');
        DB::beginTransaction();

        $address = Address::create($request->only(['post_code', 'address_line', 'area_id']));

        $supplier = Supplier::create(
            ['address_id' => $address->id] + $request->validated()
        );
        
        if ($request->has('payment_data')) {
            
            $supplier->paymentData()->create([
                'supplier_id' => $supplier->id
            ] +  $request->validated()['payment_data']);
        }
        
        if ($request->has('admins')) {
            foreach ($request->validated()['admins'] as $adminData) {
                $supplier->admins()->create($adminData);
            }
        }

        DB::commit();
        return $this->apiResponse(
            message: __('messages.supplier_created_successfully'),
            data: SupplierResource::make($supplier)
        );
    }

    public function show(Supplier $supplier)
    {
        $this->authorize('show suppliers');

        $supplier->load([
            'address',
            // 'address.area.city.country',
            'entries.journal',
            'admins',
            'paymentData',
            'paymentData.currency',
            'paymentData.country',
        ]);

        return $this->apiResponse(data: SupplierResource::make($supplier));
    }

    // public function update(UpdateSupplierRequest $request, Supplier $supplier)
    // {
    //     $this->authorize('edit suppliers');

    //     $supplier->update($request->validated());
    //     $supplier->address->update($request->only(['post_code', 'address_line', 'area_id']));

    //     return $this->apiResponse(message: __('messages.supplier_updated_successfully'), data: SupplierResource::make($supplier));
    // }
    public function update(UpdateSupplierRequest $request, Supplier $supplier)
    {
        $this->authorize('edit suppliers');
        DB::beginTransaction();
    
        $supplier->update($request->validated());
    
        if ($supplier->address) {
            $supplier->address->update($request->only(['post_code', 'address_line', 'area_id']));
        }
    
        if ($request->has('payment_data')) {
            $supplier->paymentData()->updateOrCreate(
                ['supplier_id' => $supplier->id],
                $request->validated()['payment_data']
            );
        }
    
        if ($request->has('admins')) {
            $supplier->admins()->delete(); 
            foreach ($request->validated()['admins'] as $adminData) {
                $supplier->admins()->create($adminData);
            }
        }
    
        DB::commit();
    
        return $this->apiResponse(
            message: __('messages.supplier_updated_successfully'),
            data: SupplierResource::make($supplier)
        );
    }

    public function destroy(Supplier $supplier)
    {
        $this->authorize('delete suppliers');

        $supplier->delete();
        return $this->apiResponse(message: __('messages.supplier_deleted_successfully'));
    }

    public function changeStatus(ChangeSupplierStatusRequest $request, Supplier $supplier)
    {
        $this->authorize('edit suppliers');

        $supplier->update([
            'status' => $request->status,
        ]);

        return $this->apiResponse(message: __('messages.supplier_updated_successfully'), data: SupplierResource::make($supplier));
    }
}
