<?php

namespace App\Http\Controllers;

use App\Http\Helper\CompanyHelper;
use App\Http\Requests\StoreInventoryRequest;
use App\Http\Requests\UpdateInventoryRequest;
use App\Http\Resources\InventoryResource;
use App\Http\Services\ProductInventoryService;
use App\Models\Inventory;
use App\Models\Warehouse;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class InventoryController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return $this->apiResponse(
            data: InventoryResource::collection(
                Inventory::where('company_id', CompanyHelper::getId())->get()
            )
        );
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreInventoryRequest $request)
    {
        $validatedData = $request->validated();

        DB::beginTransaction();
        $inventory = Inventory::create([
            'warehouse_id' => $validatedData['warehouse_id'],
            'company_id' => $validatedData['company_id'],
            'date' => $validatedData['date'],
            'status' => $validatedData['status'],
            'description' => $validatedData['description'],
        ]);

        foreach ($validatedData['products'] as $product) {
            $inventory->products()->create([
                'product_id' => $product['id'],
                'actual_quantity' => $product['actual_quantity'],
                'diff_quantity' => $product['diff_quantity'],
            ]);
        }
        if ($inventory->status) {
            ProductInventoryService::update($validatedData['products'], Warehouse::find($validatedData['warehouse_id']));
        }
        DB::commit();
        return $this->apiResponse(message: __('messages.inventory_created_successfully'), data: InventoryResource::make($inventory));
    }

    /**
     * Display the specified resource.
     */
    public function show(Inventory $inventory)
    {
        $inventory->load('products.product');
        return $this->apiResponse(data: InventoryResource::make($inventory));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateInventoryRequest $request, Inventory $inventory)
    {
        if ($inventory->status) {
            throw ValidationException::withMessages(['inventory' => 'can not update confirmed inventory']);
        }
        $validatedData = $request->validated();

        DB::transaction(function () use ($validatedData, $inventory) {
            $inventory->update([
                'date' => $validatedData['date'],
                'status' => $validatedData['status'],
                'description' => $validatedData['description'],
            ]);
            //delete all old products from inventory
            $inventory->products()->delete();
            //create  new  products
            foreach ($validatedData['products'] as $product) {
                $inventory->products()->create([
                    'product_id' => $product['id'],
                    'actual_quantity' => $product['actual_quantity'],
                    'diff_quantity' => $product['diff_quantity'],
                ]);
            }
            if ($inventory->status) {
                ProductInventoryService::update($validatedData['products'], Warehouse::find($inventory->warehouse_id));
            }
        });
        return $this->apiResponse(message: __('messages.inventory_updated_successfully'));
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Inventory $inventory)
    {
        $inventory->products()->delete();
        $inventory->delete();
        return $this->apiResponse(message: __('messages.inventory_deleted_successfully'));
    }
}
