<?php

namespace App\Http\Requests\Api\Product;

use App\Models\Product;
use App\Models\ProductUnit;
use Illuminate\Validation\Rule;
use App\Models\ProductWarehouse;
use App\Http\Helper\CompanyHelper;
use Illuminate\Database\Query\Builder;
use App\Http\Enums\Product\ProductTypeEnum;
use App\Models\ProductBranch;
use App\Rules\UniqueBarcodePerCompany;
use Illuminate\Foundation\Http\FormRequest;

class UpdateProductRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array|string>
     */
    public function rules(): array
    {
        $product = $this->route('product');

        return
            array_merge(
                array_merge(
                    Product::RULES,
                    [
                        'product_details.key' =>
                        [
                            'nullable',
                            Rule::unique('products')->ignore($this->route()->product->id)->where(
                                fn(Builder $query) => $query->where('company_id', CompanyHelper::getId())
                            ),
                        ],
                        'product_details.type' => 'nullable|in:' . implode(',', ProductTypeEnum::getTypes()),
                    ]
                ),


                array_merge(
                    ProductUnit::RULES,
                    [
                        "product_unit.*.barcode" => [
                            'nullable',
                            'string',
                            new UniqueBarcodePerCompany(CompanyHelper::getId(), $product->id)
                        ],
                    ]
                ),
                ProductWarehouse::RULES,
                ProductBranch::RULES
            );
    }

    public function prepareForValidation()
    {
        $this->merge([
            'product_details' => array_merge($this->input('product_details') ?? [], [
                'has_expiry_date' => ($this->product_details['has_expiry_date'] ??  'off') == 'on',
                'name' => $this->prepareName(),
                'description' => $this->prepareDescription(),
                'weight' => 0
            ]),
        ]);
    }

    private function prepareName(): array
    {
        $name = [];

        $name['ar'] = isset($this->product_details['name_ar']) ? $this->product_details['name_ar'] : null;

        if (isset($this->product_details['name_en'])) {
            $name['en'] =  $this->product_details['name_en'];
        } elseif (isset($this->product_details['name_ar'])) {
            $name['en'] =  $this->product_details['name_ar'];
        } else {
            $name['en'] =  null;
        }

        return $name;
    }

    private function prepareDescription(): array
    {
        return [
            'ar' => isset($this->product_details['description_ar']) ? $this->product_details['description_ar'] : null,
            'en' => isset($this->product_details['description_en']) ? $this->product_details['description_en'] : null,
        ];
    }
}
