<?php

namespace App\Http\Requests\Api\StockTransfer;

use App\Models\Warehouse;
use App\Models\ProductWarehouse;
use Illuminate\Validation\Validator;
use Illuminate\Foundation\Http\FormRequest;

class StoreStockTransferRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        return [
            'product_id' => 'required|exists:products,id',

            'from_warehouse' => 'required|exists:warehouses,id',
            'to_warehouse' => 'required|exists:warehouses,id',

            'from_branch' => 'required|exists:branches,id',
            'to_branch' => 'required|exists:branches,id',

            'quantity' => 'required|integer|min:1',
        ];
    }

    public function withValidator(Validator $validator)
    {
        $validator->after(function ($validator) {
            if (!$this->validateFromWarehouseBelongsToBranch()) {
                $validator->errors()->add(
                    'from_warehouse',
                    'From warehouse does not belong to branch!'
                );
            }

            if (!$this->validateToWarehouseBelongsToBranch()) {
                $validator->errors()->add(
                    'to_warehouse',
                    'To warehouse does not belong to branch!'
                );
            }

            // if (!$this->validateProductBelongsToBothBranches()) {
            //     $validator->errors()->add(
            //         'product_id',
            //         'Product does not belong to one of the two warehouses!'
            //     );
            // }
        });
    }

    private function validateFromWarehouseBelongsToBranch(): bool
    {
        return Warehouse::findOrFail($this->from_warehouse)->branches()->where('branch_id', $this->from_branch)->exists();
    }

    private function validateToWarehouseBelongsToBranch(): bool
    {
        return Warehouse::findOrFail($this->to_warehouse)->branches()->where('branch_id', $this->to_branch)->exists();
    }

    private function validateProductBelongsToBothBranches(): bool
    {
        return
            ProductWarehouse::where(
                [
                    'product_id' => $this->product_id,
                    'warehouse_id' => $this->from_warehouse
                ]
            )->exists()
            &&
            ProductWarehouse::where(
                [
                    'product_id' => $this->product_id,
                    'warehouse_id' => $this->to_warehouse
                ]
            )->exists();
    }
}
