<?php

namespace App\Http\Services\Account;

use App\Http\Helper\AccountHelper;
use App\Models\Level;
use App\Models\Account;
use App\Http\Helper\CompanyHelper;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Validation\ValidationException;


class AccountService
{
    public static Level $nextLevel;

    public static function generateCode($parentAccount = null): string
    {
        $parentCode = $parentAccount ? $parentAccount->code : ''; // 1

        static::$nextLevel = static::findAccountNextLevel($parentAccount); // level id =6

        // return self::generateFullCode(self::getAccountOrder($parentAccount), static::$nextLevel->digits) . $parentCode;
        return $parentCode . '' . self::getAccountOrder($parentAccount);
    }

    private static function getAccountOrder(?Account $parentAccount = null): string
    {
        if (is_null($parentAccount?->id)) {
            return (string)(CompanyHelper::getCompany()->accounts()->whereNull('parent_id')->count() + 1);
        }

        return (string) ($parentAccount->children()->count() + 1);
    }

    private static function generateFullCode(string $code, int $levelDigits): string
    {
        return str_pad($code, $levelDigits, '0', STR_PAD_RIGHT);
    }

    private static function findAccountNextLevel(?Account $parentAccount = null): Level
    {
        if (is_null($parentAccount?->level_id)) {
            return Level::select(['id', 'digits'])->first();
        }

        return Level::where('id', $parentAccount->level_id + 1)->select(['id', 'digits'])->firstOr(
            function () {
                throw ValidationException::withMessages(['parent_id' => [__('messages.cannot_add_deeper_level')]]);
            }
        );
    }

    // public static function deleteAccount($account): void
    // {
    //     if ($account->entries()->first()) {
    //         throw ValidationException::withMessages(['account' => [__('messages.cannot_delete_account_with_entries')]]);
    //     }
    //     if (strlen($account->code) <= 2) {
    //         throw ValidationException::withMessages([
    //             'account' => __('messages.delete_account_less_than_level_three')
    //         ]);
    //     }
    //     if ($account->children()->exists()) {
    //         throw ValidationException::withMessages(['account' => [__('messages.account_has_children')]]);
    //     }
    //     $account = Account::withTrashed()->findOrFail($account->id);
    //     if ($account->status == "sub" && $account->entries->isEmpty()) {
    //         // If the account status is "sub" and it has no entries, force delete it
    //         $account->forceDelete();
    //     } elseif ($account->trashed()) {
    //         // If the account is already soft deleted, force delete it
    //         $account->forceDelete();
    //     } else {
    //         // Otherwise, soft delete the account
    //         $account->delete();
    //     }
    //     if ($account->parent) {
    //         self::reOrderChildrenForParentAccount($account);
    //     }
    // }
    public static function deleteAccount($account, $forceDelete = false): void
    {
        // Validate the account before deletion
        if ($account->entries()->exists()) {
            throw ValidationException::withMessages(['account' => [__('messages.cannot_delete_account_with_entries')]]);
        }
    
        if (strlen($account->code) <= 2) {
            throw ValidationException::withMessages([
                'account' => __('messages.delete_account_less_than_level_three'),
            ]);
        }
    
        if ($account->children()->exists()) {
            throw ValidationException::withMessages(['account' => [__('messages.account_has_children')]]);
        }
    
        // Retrieve the account, including trashed if applicable
        $account = Account::withTrashed()->findOrFail($account->id);
    
        // Determine deletion type
        if ($forceDelete || ($account->trashed() && $account->entries->isEmpty())) {
            // Force delete if explicitly requested or already soft-deleted with no entries
            $account->forceDelete();
        } else {
            // Otherwise, perform a soft delete
            $account->delete();
        }
    
        // Reorder parent account's children if applicable
        if ($account->parent) {
            self::reOrderChildrenForParentAccount($account);
        }
    }

    public static function getAccountForModel($model, $group, ?int $company_id = null): Account
    {
        $account = Account::where('company_id',  $company_id ?? CompanyHelper::getCompany(request())->id)
            ->whereHas('accountModelGroups', function (Builder $query) use ($model, $group) {
                $query->whereHas('mainModel', function (Builder $query) use ($model) {
                    $query->where('name', $model);
                })
                    ->whereHas('modelGroup', function (Builder $query) use ($group) {
                        $query->where('name->en', $group);
                    });
            })
            ->first();
       
        if (!$account) {
            throw ValidationException::withMessages([
                'account' => [__('messages.provide_account', ['model' => AccountHelper::getNameForModel($model)])]
            ]);
        }

        return $account;
    }

    // public static function updateMultiAccountAmountsWithParentAccounts($accounts_with_amount_types): float
    // {
    //     $total = 0;

    //     foreach ($accounts_with_amount_types as $accounts_with_amount_type) {
    //         $operator = self::getOperator(account: $accounts_with_amount_type['account'], entry_type: $accounts_with_amount_type['entry_type']);
    //         self::updateAccountAmountWithParentAccounts(
    //             account: $accounts_with_amount_type['account'],
    //             amount: $accounts_with_amount_type['amount'],
    //             operator: $operator,
    //         );
    //         $total += $accounts_with_amount_type['amount'];
    //         // $total = self::getTotal($total, $accounts_with_amount_type['amount'], $operator);
    //     }
    //     return $total;
    // }
    public static function updateMultiAccountAmountsWithParentAccounts($accounts_with_amount_types): float
    {
        $total = 0;

        foreach ($accounts_with_amount_types as $accounts_with_amount_type) {
            $account = $accounts_with_amount_type['account'];

            if ($account instanceof \App\Models\Account) {
                $account = \App\Models\Account::find($account->id);
            }

            if ($account instanceof \Illuminate\Database\Eloquent\Relations\HasOneThrough) {
                $account = $account->first();
            }

            $operator = self::getOperator(
                account: $account,
                entry_type: $accounts_with_amount_type['entry_type']
            );
            self::updateAccountAmountWithParentAccounts(
                account: $account,
                amount: $accounts_with_amount_type['amount'],
                operator: $operator
            );
            $total += $accounts_with_amount_type['amount'];
        }

        return $total;
    }

    public static function getOperator(?Account $account, $entry_type): string
    {
        return $account?->type === $entry_type ? '+' : '-';
    }

    public static function updateAccountAmountWithParentAccounts(?Account  $account, $amount, $operator = '+'): ?Account
    {
        $newAmount = self::getNewAmount($account?->amount, $amount, $operator);

        $account?->update([
            'amount' =>  $newAmount,
        ]);

        self::updateParentAccounts($account, $amount, $operator);

        return $account;
    }


    private static function updateParentAccounts(?Account $account, $amount, $operator): void
    {
        $parent = $account?->parent;

        while ($parent) {

            $parent->update([
                'amount' => self::getNewAmount($parent->amount, $amount, $operator),
            ]);

            $parent = $parent->parent;
        }
    }
    private static function getNewAmount($currentAmount = 0, $amount = 0, $operator): float
    {
        $new_amount = ($operator === '-') ? ($currentAmount - $amount) : ($currentAmount + $amount);

        return $new_amount;
    }


    public static function transferAccount(Account $account, Account $parent)
    {
        if ($parent->status === 'sub') {
            throw ValidationException::withMessages([
                'account' => __('messages.account_transfer_to_sub_parent')
            ]);
        }
        if ($account->parent->code == $parent->code) {
            return;
        }

        //first update amount of old parent account
        self::updateParentAccounts($account, $account->amount, '-');

        //update the new account code for new parent based on children of new parent
        //assign the new account to new parent account
        $lastChild = $parent->children()->orderBy('code', 'desc')->first();
        $account->update([
            'code' => self::updateCode($lastChild, $parent),
            'parent_id' => $parent->id,
        ]);
        $account->load('parent');
        //reorder the children code for  old parent account
        self::reOrderChildrenForParentAccount($account);
        //add the amount to new parent account
        self::updateParentAccounts($account, $account->amount, '+');
    }
    private static function reOrderChildrenForParentAccount(Account $account)
    {
        $parentChildren = $account->parent->children()->where('code', '>', $account->code)->get();
        foreach ($parentChildren as $parentChild) {
            $parentChild->update(['code' => self::updateCode(account: $parentChild, operator: '-')]);
        }
    }

    private static function updateCode(?Account $account, ?Account $parent = null,  $operator = '+')
    {
        if (is_null($account)) {
            return self::generateCode($parent);
        }
        $newCode =  $operator === '+' ? ((int)$account->code + 1) : ((int)$account->code - 1);
        return (string) $newCode;
    }
}
