<?php

namespace App\Http\Services\Account;

use App\Http\Helper\CompanyHelper;
use App\Models\Branch;
use App\Models\JournalRepeationNotification;
use App\Models\JournalReptation;
use App\Models\Warehouse;
use Carbon\Carbon;
use Illuminate\Validation\ValidationException;


class JournalRepeationService extends JournalService
{
    public static function getNotifications()
    {
        $company_id = CompanyHelper::getId();
        $journalRepeations = JournalReptation::where('company_id', $company_id)->get();
        foreach ($journalRepeations as $journalRepeation) {

            while (self::checkDateDifference($journalRepeation)) {
                self::updateUpdatedAtBasedOnPeriod($journalRepeation);
                JournalRepeationNotification::create(
                    [
                        'company_id' => $journalRepeation->company_id,
                        'journal_repeation_id' => $journalRepeation->id,
                        'date' => $journalRepeation->updated_at
                    ]
                );
            }
        }

        return JournalRepeationNotification::where('company_id', $company_id)->where('action', null)
            ->where('date', '<=', Carbon::now())
            ->get();
    }

    private static function checkDateDifference($journalRepeation): bool
    {
        $startDate = null;
        $now = Carbon::now()->toDateString();
        $end = Carbon::parse($journalRepeation->end_date);


        //for first Time check
        if ($journalRepeation->created_at == $journalRepeation->updated_at) {
            $startDate = $journalRepeation->start_date;
        } else {
            $startDate = $journalRepeation->updated_at;
            //end date comes
            if ($end->toDateString() <= $journalRepeation->updated_at) {

                return false;
            }
        }

        $start = Carbon::parse($startDate);

        switch ($journalRepeation->period) {
            case 1:
                // Daily
                return $start->diffInDays($now) >= 1;
            case 2:
                // Weekly
                return $start->diffInWeeks($now) >= 1;
            case 3:
                // Monthly
                return $start->diffInMonths($now) >= 1;
            case 4:
                // Every 3 Months
                return $start->diffInMonths($now) >= 3;
            case 5:
                // Every 6 months
                return $start->diffInMonths($now) >= 6;
            case 6:
                // Every year
                return $start->diffInYears($now) >= 1;
            default:
                return false;
        }
    }

    private static function updateUpdatedAtBasedOnPeriod($journalRepeation)
    {
        $dateToBeAdded = null;

        if ($journalRepeation->created_at == $journalRepeation->updated_at) {
            $dateToBeAdded = Carbon::parse($journalRepeation->start_date);
        } else {
            $dateToBeAdded = Carbon::parse($journalRepeation->updated_at);
        }
        switch ($journalRepeation->period) {
            case 1:
                // Add 1 day
                $dateToBeAdded->addDay();
                break;
            case 2:
                // Add 1 week
                $dateToBeAdded->addWeek();
                break;
            case 3:
                // Add 1 month
                $dateToBeAdded->addMonth();
                break;
            case 4:
                // Add 6 months
                $dateToBeAdded->addMonths(6);
                break;
            case 5:
                // Add 1 year
                $dateToBeAdded->addYear();
                break;
            default:
                break;
        }
        $journalRepeation->timestamps = false;

        $journalRepeation->updated_at = $dateToBeAdded;

        $journalRepeation->save();

        $journalRepeation->timestamps = true;
    }

    public static function handelNotification($request)
    {

        $journalNotification = JournalRepeationNotification::find($request->id);
        if ($request->action) {
            $debit_side = JournalService::handleSide(
                JournalService::filterAccountsSide(
                    accounts: $request->accounts,
                    side: 'debit'
                ),
                'debit'
            );

            $credit_side = JournalService::handleSide(
                JournalService::filterAccountsSide(
                    accounts: $request->accounts,
                    side: 'credit'
                ),
                'credit'
            );

            if ($debit_side['side_amount'] !== $credit_side['side_amount']) {
                throw ValidationException::withMessages(['debit' => [__('messages.journal_validation_error')]]);
            }
            JournalService::createJournal(
                date: $request->date ?? now(),
                type: __('constants.journal_manual_type'),
                source: 'Manual',
                description: $request->description,
                status: $request->status,
                debit: $debit_side['side_data'],
                credit: $credit_side['side_data'],
                reference_code: $request->reference_code,
                branch: Branch::findOr($request->branch_id, fn() => null),
                warehouse: Warehouse::findOr($request->warehouse_id, fn() => null),
                repeatable: null,
                repeatableData: null,
                employee: null,
                file: null,
            );
        }

        $journalNotification->action = $request->action;
        $journalNotification->save();
        return $request->action;
    }

    public static function repeatJournal($request)
    {
        $journalNotification = JournalRepeationNotification::find($request->id);
        if ($request->action) {
            self::duplicateJournal($journalNotification->journalRepeation->journal);
        }

        $journalNotification->action = $request->action;
        $journalNotification->save();
        return $request->action;
    }
}
