<?php

namespace App\Http\Services\Bill;

use App\Models\Bill;
use App\Models\Service;
use App\Trait\BillTrait;
use App\Models\Warehouse;
use App\Models\BillService;
use App\Models\ProductUnit;
use App\Models\DismissalNotice;
use App\Models\ProductServiceUnit;
use Illuminate\Validation\ValidationException;

class BillServiceService
{
    use BillTrait;

    private array $requestedServicesData;
    public array $billProcessedItemsData = [];

    private array $services;
    private array $dismissalNoticesData = [];
    protected Bill $bill;
    protected Warehouse $warehouse;

    public $total = 0;
    public $cost = 0;
    public $discount = 0;
    public $totalDiscountOnTheBill = 0;
    public $tax = 0;

    public function __construct(array $requestedServicesData, Warehouse $warehouse)
    {
        $this->requestedServicesData = $requestedServicesData;
        $this->warehouse = $warehouse;
    }

    public function updateDismissalNotices(): self
    {
        foreach ($this->dismissalNoticesData as $dismissalNotice) {
            $total_used_quantities = array_sum($dismissalNotice['used_quantity']);

            $newQuantity = $dismissalNotice['dismissalNotice']->use_quantity - $total_used_quantities;

            if ($newQuantity < 0) {
                throw ValidationException::withMessages([
                    'service_id' => [__('messages.service_quantity_validation_message')]
                ]);
            }

            $dismissalNotice['dismissalNotice']->decrement('use_quantity', $total_used_quantities * 100);
        }

        return $this;
    }

    public function processRequestedServices(): self
    {
        $this->cost = 0;
        $this->total = 0;
        $this->discount = 0;
        $this->tax = 0;

        $currentServiceIds = isset($this->bill) ? $this->getBillServicesIds() : [];

        foreach ($this->requestedServicesData as $key => $billService) {
            $service = self::getServiceDetails($billService);


            // $this->checkProductsHaveAvailableDismissalNotices(
            //     productIds: $service->productServiceUnits->pluck('product_id')->toArray(),
            //     branchId: $billService['branch_id'],
            // );

            $discount = $this->calcDiscount(
                discount_type: $billService['discount_type'],
                discount_amount: $billService['discount_amount'],
                price: $billService['price']
            );

            $total = $this->calcProductsCost(
                quantity: $billService['quantity'],
                service: $service
            )
                ->calcTotal(
                    price: $billService['price'],
                    quantity: $billService['quantity'],
                    discount: $discount
                );

            $processedService['item'] = $service;
            $processedService['quantity'] = $billService['quantity'];
            $processedService['price'] = $billService['price'];
            $processedService['discount'] = $discount;
            $processedService['total'] = $total;

            // calc tax
            if (isset($billService['tax_id'])) {

                $this->calcTax($service->price, $this->tax($billService['tax_id']), $billService['tax_included'], $billService['quantity']);


                $processedService['tax_id'] = $this->tax($billService['tax_id']);
                $processedService['tax_included'] = $billService['tax_included'];
            }

            $this->addProcessedItem($processedService);

            $this->services[] = $service;

            if (isset($this->bill)) {
                $this->totalDiscountOnTheBill += $this->billProcessedItemsData[$key]['from_total_discount'];
                $this->processBillService(service: $service, billService: ['from_total_discount' => $this->billProcessedItemsData[$key]['from_total_discount']] + $billService);
            }

            unset($currentServiceIds[array_search($service->id, $currentServiceIds)]);
        }

        isset($this->bill) ? $this->deleteUnwantedServices($currentServiceIds) : null;

        return $this;
    }

    private function processBillService(Service $service, array $billService): BillService
    {
        return BillService::updateOrCreate([
            'bill_id' => $this->bill->id,
            'service_id' => $service->id,
        ], [
            'branch_id' => $billService['branch_id'],
            'discount_type' => $billService['discount_type'],
            'from_total_discount' => $billService['from_total_discount'],
            'discount_amount' => $billService['discount_amount'],
            'date' => $billService['date'],
            'quantity' => $billService['quantity'],
            'price' => $billService['price'],
            'starts_at' => $billService['starts_at'],
            'tax_included' => $billService['tax_included'],
            'tax_id' => isset($billService['tax_id']) ? $billService['tax_id'] : null,
        ]);
    }

    private function checkProductsHaveAvailableDismissalNotices(array $productIds, int $branchId): self
    {
        foreach ($productIds as $productId) {
            $dismissalNotice = DismissalNotice::where([
                'product_id' => $productId,
                'branch_id' => $branchId,

            ])
                ->orderBy('use_quantity', 'desc')
                ->first();

            if (!$dismissalNotice) {
                throw ValidationException::withMessages([
                    'service_id' => [__('messages.dismissal_note_required_message')]
                ]);
            }

            $this->dismissalNoticesData[$dismissalNotice->product_id]['dismissalNotice'] = $dismissalNotice;
        }

        return $this;
    }

    private function calcProductsCost(int $quantity, Service $service): self
    {
        foreach ($service->productServiceUnits as $productServiceUnit) {
            $this->cost += ($this->calcServiceCost($productServiceUnit, $quantity));
        }

        return $this;
    }

    private function calcTotal(float $price, int $quantity, float $discount): float
    {
        $total = $price * $quantity;

        if ($discount > $total) {
            throw ValidationException::withMessages([
                'discount' => [__('messages.discount_greater_than_total_message')]
            ]);
        }

        $this->total += $total;

        return $total;
    }

    private function calcServiceCost(ProductServiceUnit $productServiceUnit, int $quantity): float
    {
        $serviceProductUnit = ProductUnit::where([
            'unit_id' => $productServiceUnit?->company_unit_id,
            'product_id' => $productServiceUnit?->product_id,
        ])->first();

        $conversion_factor = $serviceProductUnit ? $serviceProductUnit->conversion_factor : 1;

        $productCostPrice = $productServiceUnit?->product->warehouses->first()->price;

        $used_quantity = ($productServiceUnit->percentage / 100) * $quantity * $conversion_factor;

        $cost = $used_quantity * $productCostPrice;

        $this->dismissalNoticesData[$productServiceUnit?->product_id]['used_quantity'][$productServiceUnit->service_id] = $used_quantity;

        return $cost;
    }

    private static function getServiceDetails(array $billService): Service
    {
        return Service::where([
            'id' => $billService['service_id'],
        ])
            ->whereHas('branches', fn($q) => $q->where('branch_id', $billService['branch_id']))
            // ->whereHas('productServiceUnits.product.warehouses', fn ($q) => $q->where('warehouse_id', $billService['warehouse_id']))
            ->with([
                'productServiceUnits:id,product_id,service_id,company_unit_id,percentage',
                'productServiceUnits.product:id',
                'productServiceUnits.product.warehouses'
                => fn($q) => $q->where('warehouse_id', $billService['warehouse_id'])
                    ->select('id', 'price', 'quantity', 'value', 'product_id')
            ])
            ->firstOrFail(['id', 'price']);
    }
}
