<?php

namespace App\Http\Services\DismissalNotice;

use App\Models\Branch;
use App\Models\Journal;
use App\Models\Product;
use App\Models\Warehouse;
use App\Models\DismissalNotice;
use App\Models\ProductWarehouse;
use App\Http\Helper\AccountHelper;
use Illuminate\Support\Facades\DB;
use App\Http\Services\Account\AccountService;
use App\Http\Services\Account\JournalService;
use Illuminate\Validation\ValidationException;

class DismissalNoticeService
{
    public static function createDismissal(array $data): DismissalNotice
    {
        try {
            DB::beginTransaction();
            $productWarehouse = self::getProductWarehouse(
                productId: $data['product_id'],
                warehouseId: $data['warehouse_id']
            );

            throw_if(
                !self::quantityAvailable(productWarehouse: $productWarehouse, quantity: $data['quantity']),
                ValidationException::withMessages([
                    'quantity' => [__('messages.quantity_not_enough_in_warehouse')]
                ])
            );

            $dismissalNotice = self::createDismissalNotice(['use_quantity' => $data['quantity']] + $data);

            self::createDismissalNoticeJournal(
                amount: $productWarehouse->price *  $data['quantity'],
                product: $productWarehouse->product,
                dismissalNotice: $dismissalNotice,
                branch: Branch::findOr($data['branch_id'], fn () => null),
                warehouse: Warehouse::findOr($data['warehouse_id'], fn () => null),
            );

            self::updateWarehouseQuantity(productWarehouse: $productWarehouse, quantity: $data['quantity']);

            DB::commit();

            return $dismissalNotice;
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    public static function createDismissalNotice(array $data): DismissalNotice
    {
        return DismissalNotice::create($data);
    }

    public static function createDismissalNoticeJournal(
        $amount,
        Product $product,
        DismissalNotice $dismissalNotice,
        ?Branch $branch = null,
        ?Warehouse $warehouse = null,
    ) {
        JournalService::createJournal(
            date: now(),
            type: __('constants.auto_type'),
            source: 'Dismissal Notice',
            description: __('messages.dismissal_notice_description'),
            file: null,
            employee: null,
            status: true,
            debit: [
                [
                    'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_SUPPLIES_WAREHOUSE),
                    'type' => [
                        'type' => DismissalNotice::class,
                        'id' => $dismissalNotice->id,
                    ],
                    'amount' => $amount,
                    'entry_type' => "debit",
                ],
            ],
            credit: [
                [
                    'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_PRODUCTS),
                    'type' => [
                        'type' => Product::class,
                        'id' => $product->id,
                    ],
                    'amount' => $amount,
                    'entry_type' => "credit",
                ],
            ],
            branch: $branch ?? null,
            warehouse: $warehouse ?? null,
            journalable: $dismissalNotice
        );
    }

    public static function getProductWarehouse($productId, $warehouseId): ProductWarehouse
    {
        return ProductWarehouse::where([
            'warehouse_id' => $warehouseId,
            'product_id' => $productId,
        ])->with(['product:id'])
            ->firstOr(['quantity', 'value', 'price', 'product_id'], function () {
                throw ValidationException::withMessages([
                    'warehouse_id' => [__('messages.product_not_in_warehouse')]
                ]);
            });
    }

    public static function quantityAvailable(ProductWarehouse $productWarehouse, $quantity): bool
    {
        return $productWarehouse->quantity >= $quantity;
    }

    public static function updateWarehouseQuantity(ProductWarehouse $productWarehouse, $quantity): void
    {
        $warehouse_new_quantity = $productWarehouse->quantity - $quantity;

        $productWarehouse->update([
            'quantity' => $warehouse_new_quantity,
            'value' => $warehouse_new_quantity * $productWarehouse->price,
        ]);
    }
}
