<?php

namespace App\Http\Services\StockTransfer;

use App\Models\Branch;
use App\Models\Journal;
use App\Models\Product;
use App\Models\Warehouse;
use App\Models\ProductUnit;
use App\Models\StockTransfer;
use App\Models\ProductWarehouse;
use App\Http\Helper\AccountHelper;
use App\Jobs\ProductTrackQuantity;
use Illuminate\Support\Facades\DB;
use App\Http\Services\Account\AccountService;
use App\Http\Services\Account\JournalService;
use Illuminate\Validation\ValidationException;
use App\Http\Requests\Api\StockTransfer\StoreStockTransferRequest;

class StockTransferService
{
    private ProductWarehouse $fromProductWarehouse;
    private ProductWarehouse $toProductWarehouse;
    private StockTransfer $stockTransfer;
    private StoreStockTransferRequest $request;

    private int $productId;
    private int $quantity;

    public function processTransfer(StoreStockTransferRequest $request): self
    {
        return
            DB::transaction(function () use ($request) {
                return $this->setRequest(request: $request)
                    ->updateWarehousesQuantity()
                    ->createStockTransfer()
                    ->createStockTransferJournals();
            });
    }

    private function updateWarehousesQuantity(): self
    {
        $this->updateFromWarehouse()->updateToWarehouse();

        return $this;
    }

    private function updateFromWarehouse(): self
    {
        $fromProductWarehouse = ProductWarehouse::where([
            'warehouse_id' => $this->request->from_warehouse,
            'product_id' => $this->productId,
        ])->firstOrFail();

        if ($fromProductWarehouse->quantity < $this->quantity) {
            throw ValidationException::withMessages([
                'from_warehouse' => [__('messages.quantity_not_enough_in_warehouse')]
            ]);
        }

        $fromQuantity = $this->calcFromQuantity(fromProductWarehouse: $fromProductWarehouse);

        $fromProductWarehouse->update([
            'quantity' =>  $fromQuantity,
            'value' =>  $this->calcFromValue(fromProductWarehouse: $fromProductWarehouse, fromQuantity: $fromQuantity),
        ]);

        $this->fromProductWarehouse = $fromProductWarehouse;

        return $this;
    }

    private function calcFromValue(ProductWarehouse $fromProductWarehouse, int $fromQuantity): int
    {
        return $fromQuantity / ($fromProductWarehouse->price ? $fromProductWarehouse->price : 1);
    }

    private function calcFromQuantity(ProductWarehouse $fromProductWarehouse): int
    {
        return $fromProductWarehouse->quantity - $this->quantity;
    }

    private function updateToWarehouse(): self
    {
        $warehouseData =
            [
                'warehouse_id' => $this->request->to_warehouse,
                'product_id' => $this->productId,
            ];

        $toProductWarehouse = ProductWarehouse::where($warehouseData)->firstOr(fn () => ProductWarehouse::create($warehouseData));

        $toValue = $this->calcToValue(toProductWarehouse: $toProductWarehouse);
        $toQuantity = $this->calcToQuantity(toProductWarehouse: $toProductWarehouse);

        $toProductWarehouse->update(
            [
                'quantity' =>  $toQuantity,
                'value' =>  $toValue,
                'price' =>  $toValue / $toQuantity,
            ]
        );

        $this->toProductWarehouse = $toProductWarehouse;

        return $this;
    }

    private function calcToValue(ProductWarehouse $toProductWarehouse): float
    {
        return $toProductWarehouse->value + ($this->quantity * ($this->fromProductWarehouse->price ? $this->fromProductWarehouse->price : 1));
    }

    private function calcToQuantity(ProductWarehouse $toProductWarehouse): int
    {
        return $toProductWarehouse->quantity + $this->quantity;
    }

    private function createStockTransferJournals(): self
    {
        $total = $this->fromProductWarehouse->price * $this->quantity;

        $journal = JournalService::createJournal(
            date: now(),
            type: __('constants.auto_type'),
            source: 'Stock Transfer',
            description: __('messages.stock_transfer_journal_description'),
            file: null,
            employee: null,
            status: true,
            debit: $this->debitSide(total: $total),
            credit: $this->creditSide(total: $total),
            journalable: $this->stockTransfer,
            branch: Branch::findOr($this->request->branch_id, fn () => null),
            warehouse: Warehouse::findOr($this->request->warehouse_id, fn () => null),
        );

        $product = Product::select(['id'])->find($this->productId);
        $productUnit = ProductUnit::where(['product_id' => $this->productId, 'conversion_factor' => 1])->first(['id']);

        ProductTrackQuantity::dispatch(
            product: $product,
            journal: $journal,
            productUnit: $productUnit,
            quantity: -$this->quantity,
            productWarehouse: $this->fromProductWarehouse,
        );

        ProductTrackQuantity::dispatch(
            product: $product,
            journal: $journal,
            productUnit: $productUnit,
            quantity: $this->quantity,
            productWarehouse: $this->toProductWarehouse,
        );

        return $this;
    }

    private function creditSide(float $total): array
    {
        return [
            [
                'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_PRODUCTS),
                'type' => [
                    'type' => Product::class,
                    'id' => $this->productId,
                ],
                'amount' => $total,
                'entry_type' => 'credit',
            ],
        ];
    }

    private function debitSide(float $total): array
    {
        return [
            [
                'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_PRODUCTS),
                'type' => [
                    'type' => Product::class,
                    'id' => $this->productId,
                ],
                'amount' => $total,
                'entry_type' => 'debit',
            ],
        ];
    }

    private function setData(): self
    {
        $this->productId = $this->request->product_id;
        $this->quantity = $this->request->quantity;
        return $this;
    }

    private function createStockTransfer(): self
    {
        $this->stockTransfer = StockTransfer::create($this->request->validated());
        return $this;
    }

    private function setRequest(StoreStockTransferRequest $request): self
    {
        $this->request = $request;
        $this->setData();
        return $this;
    }
}
