<?php

namespace App\Imports;

use App\Http\Enums\Product\ProductInventoryTypeEnum;
use App\Http\Enums\Product\ProductTypeEnum;
use App\Http\Helper\CompanyHelper;
use App\Http\Services\ProductsImportService;
use App\Models\Branch;
use App\Models\Category;
use App\Models\CompanyUnit;
use App\Models\Product;
use App\Models\ProductUnit;
use App\Models\ProductWarehouse;
use App\Models\Supplier;
use App\Models\Tax;
use App\Models\Warehouse;
use App\Models\WarehouseUnit;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithStartRow;


class ProductsImport implements ToModel, WithStartRow
{

    public $productWarehouseId;
    public $unitId;
    public $purchasing_price;
    public $selling_price;
    private $productPurchasingPrice;

    public function __construct(private $productService)
    {
        // Initialize the service
        $this->productService = $productService;
    }


    public function startRow(): int
    {
        return 2;
    }
    /**
     * @param array $row
     *
     * @return \Illuminate\Database\Eloquent\Model|null
     */
    public function model(array $row)
    {
        $wareHouseBranchName = explode(' - ', $row[8]);
        $warehouseId = $this->getWarehouseId($wareHouseBranchName[0]);
        $this->unitId = $this->getUnitId($row[6]);
        $product = new Product([
            'code' => $row[0],
            'key' => $row[1],
            'category_id' => $this->getCategoryId($row[5]),
            'tax_id' => $this->getTaxId($row[12]),
            'company_id' => CompanyHelper::getId(),
            'inventory_type' => $row[19] === null ? null : ProductInventoryTypeEnum::getValueFromArabicType($row[19]),
            'type' => $row[20] === null ? null : ProductTypeEnum::getValueFromArabicType($row[20]),
            'has_expiry_date' => $this->getStatusFromText($row[21]),
        ]);

        $product->setTranslations('name', [
            'ar' => $row[3],
            'en' => $row[3],
        ]);
        $product->setTranslations('description', [
            'ar' => $row[4],
            'en' => $row[4],
        ]);

        $product->save();
        $this->storeProductUnit($product, $this->unitId, $row);
        $this->storeProductWarehouse($product, $warehouseId, $row);
        $this->storeWarehouseUnit($this->productWarehouseId, $this->unitId, $this->purchasing_price, $this->selling_price);

        return $product;
    }


    private function getCategoryId($categoryName)
    {
        return Category::where('name->ar', $categoryName)
            ->orWhere('name->en', $categoryName)
            ->where('company_id', CompanyHelper::getId())
            ->pluck('id')
            ->first();
    }

    private function getTaxId($taxRate)
    {
        return Tax::where('rate', $taxRate)
            ->where('company_id', CompanyHelper::getId())
            ->pluck('id')
            ->first();
    }
    private function getSupplierId($supplierName)
    {
        return Supplier::where('name->ar', $supplierName)
            ->orWhere('name->en', $supplierName)
            ->where('company_id', CompanyHelper::getId())
            ->pluck('id')
            ->first();
    }
    private function getWarehouseId($warehouseName)
    {
        return Warehouse::where('name->ar', $warehouseName)->orWhere('name->en', $warehouseName)->where('company_id', CompanyHelper::getId())->pluck('id')->first();
    }
    private function getBranchId($branchName)
    {
        return Branch::where('name', $branchName)->where('company_id', CompanyHelper::getId())
            ->orWhere('name->en', $branchName)
            ->pluck('id')
            ->first();
    }
    private function getUnitId($unitName)
    {
        return CompanyUnit::where('name->ar', $unitName)
            ->orWhere('name->en', $unitName)
            ->where('company_id', CompanyHelper::getId())
            ->orWhere('name->en', $unitName)
            ->pluck('id')
            ->first();
    }
    private function storeProductUnit(Product $product, $unitId, $row): void
    {
        $productUnit = new ProductUnit([
            'barcode' => $row[2] ?? null,
            'unit_id' => $unitId,
            'product_id' => $product->id,
            'main' => $this->getStatusFromText($row[13]),
            'weight' => 0,
            'virtual_sale' => $this->getStatusFromText($row[14]),
            'virtual_buying' => $this->getStatusFromText($row[15]),
            'conversion_factor' => $row[16] ?? 1,
            'purchasing_price' => $row[10] ?? 0,
            'selling_price' => $row[11] ?? 0,
        ]);
        $this->productPurchasingPrice = $row[10] ?? 0;
        $productUnit->save();
        $this->purchasing_price = $row[10] ?? 0;
        $this->selling_price = $row[11] ?? 0;
    }
    private function storeProductWarehouse(Product $product, $warehouseId, $row): void
    {
        $productWarehouse = new ProductWarehouse([
            'warehouse_id' => $warehouseId,
            'product_id' => $product->id,
            'quantity' => $row[10] ?? 0,
            'min_quantity' => $row[17]??0,
            'max_quantity' => $row[18] ??0,
        ]);
        $productWarehouse->save();
        $this->productWarehouseId = $productWarehouse->id;
        $this->productService->addProductsWarehousesData($warehouseId, $row[9] * $this->productPurchasingPrice);
    }
    private function storeWarehouseUnit($productWarehouseId, $unitId, $purchasing_price, $selling_price)
    {
        $warehouseUnit = WarehouseUnit::create([
            'product_warehouse_id' => $productWarehouseId,
            'unit_id' => $unitId,
            'purchasing_price' => $purchasing_price,
            'selling_price' => $selling_price,
        ]);
    }

    private function getStatusFromText($text = null): int
    {
        return $text === 'نعم' ? 1 : 0;
    }
}
