<?php

namespace App\Models;

use App\Http\Enums\Bill\BillEnum;
use App\Http\Resources\BillResource;
use App\Http\Enums\Bill\BillStatusEnum;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Bill extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'code',
        'company_id',
        'worker_id',
        'billable_type',
        'billable_id',
        'release_date',
        'amount_paid',
        'remain',
        'type',
        'before_discount',
        'total_discount',
        'tax_amount',
        'branch_id',
        'warehouse_id',
        'note',
        'status',
        'date_of_supply',
        'attachment_file',
        'total_other_expenses_amount',
        'fast_sales',
        'category',
        'total_purchase',
        'is_returned',
        //  'returned_bill'
    ];

    protected $casts = [
        'release_date' => 'date',
        'date_of_supply' => 'date',
        'status' => BillStatusEnum::class
    ];

    const FILE_PATH = 'uploads/company/bills';

    public function billable()
    {
        return $this->morphTo();
    }

    public function billExpenses()
    {
        return $this->hasMany(BillExpense::class);
    }

    public function journals()
    {
        return $this->morphMany(Journal::class, 'journalable');
    }

    public function billProducts()
    {
        return $this->hasMany(BillProduct::class);
    }

    public function billServices()
    {
        return $this->hasMany(BillService::class);
    }

    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    public function branch()
    {
        return $this->belongsTo(Branch::class);
    }

    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class);
    }

    public function delivery()
    {
        return $this->belongsTo(Delivery::class);
    }

    public function expensePaymentMethod()
    {
        return $this->hasOne(BillExpensePayment::class);
    }

    public function billPaymentMethods()
    {
        return $this->hasMany(BillPaymentMethod::class);
    }

    public function billPaymentMethod()
    {
        return $this->hasOne(BillPaymentMethod::class);
    }

    public function sellNature()
    {
        return $this->hasOne(BillSellNature::class);
    }

    public function isAccredited()
    {
        return $this->status == BillStatusEnum::ACCREDITED;
    }

    public function isSalesBill()
    {
        return $this->type == BillEnum::sales->value;
    }

    public function getProductsBeforeDiscountAttribute($value)
    {
        return $value / 100;
    }

    public function setProductsBeforeDiscountAttribute($value)
    {
        $this->attributes['products_before_discount'] = $value * 100;
    }

    public function getTotalOtherExpensesAmountAttribute($value)
    {
        return $value / 100;
    }

    public function setTotalOtherExpensesAmountAttribute($value)
    {
        $this->attributes['total_other_expenses_amount'] = $value * 100;
    }

    public function getTotalProductsDiscountAttribute($value)
    {
        return $value / 100;
    }

    public function setTotalProductsDiscountAttribute($value)
    {
        $this->attributes['total_products_discount'] = $value * 100;
    }

    public function getDiscountOnBillAttribute($value)
    {
        return $value / 100;
    }

    public function getTaxAmountAttribute($value)
    {
        return $value / 100;
    }

    public function setTaxAmountAttribute($value)
    {
        $this->attributes['tax_amount'] = $value * 100;
    }

    public function worker()
    {
        return $this->belongsTo(Worker::class);
    }
    public function getCreatedAtTimeAttribute()
    {
        return $this->created_at->format('H:i:s');
    }
    public function bounds()
    {
        return $this->belongsToMany(Bound::class, 'bound_bill')
                    ->withPivot('amount', 'discount')
                    ->withTimestamps();
    }
}
