<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Spatie\Translatable\HasTranslations;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Carbon\Carbon;

class Branch extends Model
{
    use HasFactory, SoftDeletes, HasTranslations;

    const RULES = [
        'name_ar' => 'required|string',
        'name_en' => 'nullable|string',
        'name' => 'nullable|array',
        'address_id' => 'nullable|exists:addresses,id',
        'phone_1' => 'nullable',
        'phone_2' => 'nullable',
        'area' => 'nullable|string',
        'email' => 'nullable|email',
        'website' => 'nullable',
        'delivers' => 'nullable|in:0,1',
        'group_settings_id' => 'nullable|exists:group_settings,id',
        'number_of_employees' => 'nullable|between:1,3',
        'warehouses' => 'nullable|array',
        'warehouses.*.id' => 'required|exists:warehouses,id',
        'location' => 'nullable|string',
        'status' => 'nullable|in:0,1',
        'opening_time' => 'nullable|date_format:H:i:s',
        'closing_time' => 'nullable|date_format:H:i:s|after:opening_time',
    ];

    public $translatable = [
        'name',

    ];

    protected $fillable = [
        'company_id',
        'name',
        'address_id',
        'phone_1',
        'phone_2',
        'email',
        'area',
        'website',
        'home_service',
        'beauty_expert',
        'delivers',
        'group_settings_id',
        'number_of_employees',
        'location',
        'status',
        'opening_time',
        'closing_time',
        'is_opened',
    ];
    protected  $with = ['warehouses',  'custodies', 'treasuries', 'banks'];

    public function shifts()
    {
        return $this->hasMany(Shift::class);
    }

    public function journals()
    {
        return $this->hasMany(Journal::class);
    }


    public function address()
    {
        return $this->belongsTo(Address::class, 'address_id', 'id');
    }

    public function autoShifts()
    {
        return $this->hasMany(AutoShift::class);
    }

    public function dismissalNotice()
    {
        return $this->hasMany(DismissalNotice::class);
    }

    public function deliveries()
    {
        return $this->hasMany(Delivery::class);
    }

    public function treasuries()
    {
        return $this->hasMany(Treasury::class);
    }

    public function warehouses()
    {
        return $this->belongsToMany(Warehouse::class);
    }

    public function groupSetting()
    {
        return $this->belongsTo(GroupSetting::class);
    }

    public function company()
    {
        return $this->belongsTo(Company::class);
    }



    public function services()
    {
        return $this->belongsToMany(Service::class, 'service_branches');
    }
    public function employees()
    {
        return $this->hasMany(Employee::class);
    }

    public function banks()
    {
        return $this->hasMany(Bank::class);
    }

    public function networks()
    {
        return $this->hasMany(Network::class);
    }

    public function employeeTransactions()
    {
        return $this->hasMany(EmployeeTransaction::class);
    }

    public function bills()
    {
        return $this->hasMany(Bill::class);
    }

    public function custodies()
    {
        return $this->hasMany(Custody::class);
    }

    public function isOpen()
    {
        $now = Carbon::now('Africa/Cairo')->format('H:i');
        $openingTime = Carbon::parse($this->opining_time)->format('H:i');
        $closingTime = Carbon::parse($this->closing_time)->format('H:i');
        return $now >= $openingTime && $now <= $closingTime;
    }
    public function getLocationAttribute($value)
    {

        if (preg_match('/([0-9.\-]+)!4d([0-9.\-]+)/', $value, $matches)) {

            return $matches[1] . ',' . $matches[2];
        }
        return null;
    }
    public function products()
    {
        return $this->hasMany(ProductBranch::class, 'branche_id', 'id')
        ->whereHas('product', function ($query) {
            $query->where('is_stock', 0);
        });
    }
}
