<?php

namespace App\Models;

use App\Models\BlockReason;
use Laravel\Sanctum\HasApiTokens;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Traits\HasRoles;
use Illuminate\Notifications\Notifiable;
use Spatie\Translatable\HasTranslations;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;

class Company extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable, HasTranslations, SoftDeletes, HasRoles;

    const UPLOADED_FILES = 'uploads/company';

    const ROLE = [
        'name.ar' => 'required',
        'name.en' => 'required',
        'password' => 'required',
        'lang' => 'required|in:en,ar',
        'location_id' => 'nullable|exists:locations,id,parent_id,NOT_NULL',
        'address' => 'nullable|string|max:255',
        'phone_2' => 'nullable|string|max:20',
        'CRN' => 'nullable|string|max:50',
        'TRN' => 'nullable|string|max:50',
    ];

    const PERMISSIONS_RULES = [
        'permissions' => 'nullable|array|min:1',
        'permissions.*' => 'required|exists:permissions,id',
    ];

    protected $guard = "company";

    protected $fillable = [
        'name',
        'email',
        'logo',
        'password',
        'phone',
        'lang',
        'secret_key',
        'email_verified_at',

        'location_id',
        'address',
        'phone_2',
        'steps',
        'CRN', // Commercial Registration No
        'TRN', // Tax registration number
        'steps_completed',
    ];

    public $translatable = ['name'];

    protected $hidden = [
        'password',
        'secret_key',
    ];

    public function stockTransfers()
    {
        return $this->hasManyThrough(
            StockTransfer::class,
            Product::class,
            'company_id',
            'product_id',
        );
    }

    public function autoShifts()
    {
        return $this->hasMany(AutoShift::class);
    }

    public function subscriptions()
    {
        return $this->hasMany(Subscription::class);
    }
    public function lastSubscription()
    {
        return $this->subscriptions()->active()->first();
    }
    public function companyCurrentPackageWithFeatures()
    {
        return $this->subscriptions()->active()->first()?->package()->with('features')->first();
    }

    public function orders()
    {
        return $this->hasMany(Order::class);
    }

    public function journals()
    {
        return $this->hasMany(Journal::class);
    }

    public function expenses()
    {
        return $this->hasMany(Expense::class);
    }

    public function bounds()
    {
        return $this->hasMany(Bound::class);
    }

    public function treasuries()
    {
        return $this->hasMany(Treasury::class);
    }

    public function banks()
    {
        return $this->hasMany(Bank::class);
    }

    public function networks()
    {
        return $this->hasMany(Network::class);
    }

    public function bills()
    {
        return $this->hasMany(Bill::class);
    }

   public function sellNatures()
    {
        return $this->belongsToMany(MainSellNature::class , 'company_main_sell_nature')->withPivot('label');
    }

    public function deliveries()
    {
        return $this->hasManyThrough(Delivery::class, Branch::class, 'company_id', 'branch_id', 'id', 'id');
    }

    public function paymentMethods()
    {
        return $this->belongsToMany(PaymentMethod::class);
    }

    public function shipmentCompanies()
    {
        return $this->hasMany(ShipmentCompany::class);
    }

    public function paymentTerms()
    {
        return $this->hasMany(PaymentTerm::class);
    }

    public function taxes()
    {
        return $this->hasMany(Tax::class);
    }

    public function workers()
    {
        return $this->hasManyThrough(Worker::class, Employee::class, 'company_id', 'employee_id', 'id', 'id');
    }

    public function absences()
    {
        return $this->hasManyThrough(Absence::class, Employee::class, 'company_id', 'employee_id', 'id', 'id');
    }

    public function permissionRequests()
    {
        return $this->hasManyThrough(PermissionRequest::class, Employee::class, 'company_id', 'employee_id', 'id', 'id');
    }

    public function vacationRequests()
    {
        return $this->hasManyThrough(VacationRequest::class, Employee::class, 'company_id', 'employee_id', 'id', 'id');
    }

    public function vacations()
    {
        return $this->hasMany(Vacation::class);
    }

    public function roles()
    {
        return $this->hasMany(Role::class);
    }

    public function mainSettings()
    {
        return $this->hasMany(CompanyMainSettings::class);
    }

    public function units()
    {
        return $this->hasMany(CompanyUnit::class);
    }

    public function currencies()
    {
        return $this->hasMany(CompanyCurrency::class);
    }

    public function warehouses()
    {
        return $this->hasMany(Warehouse::class);
    }

    public function suppliers()
    {
        return $this->hasMany(Supplier::class);
    }

    public function categories()
    {
        return $this->hasMany(Category::class);
    }

    public function productsCategories()
    {
        return $this->hasMany(Category::class)
            ->where('type', 'products_category');
    }

    public function servicesCategories()
    {
        return $this->hasMany(Category::class)
            ->where('type', 'services_category');
    }

    public function fastServicesCategories()
    {
        return $this->servicesCategories->where('fast', 1);
    }

    public function services()
    {
        return $this->hasMany(Service::class);
    }

    public function clients()
    {
        return $this->hasMany(Client::class);
    }

    public function clientAttributes($type, $model)
    {
        return $this->hasMany(Attribute::class)->where(['type' => $type, 'model' => $model]);
    }

    public function blockReasons()
    {
        return $this->hasMany(BlockReason::class)->where('option_label', BlockReason::KEY);
    }

    // public function inventoryTypes()
    // {
    //     return $this->hasMany(InventoryType::class)->where('option_label', InventoryType::KEY);
    // }

    public function blockClients()
    {
        return $this->hasMany(BlockClient::class);
    }

    public function blockSuppliers()
    {
        return $this->hasMany(BlockSupplier::class);
    }

    public function accounts()
    {
        return $this->hasMany(Account::class, 'company_id');
    }

    public function products()
    {
        return $this->hasMany(Product::class);
    }

    public function groupSettings()
    {
        return $this->hasMany(GroupSetting::class);
    }

    public function branches()
    {
        return $this->hasMany(Branch::class);
    }

    public function careers()
    {
        return $this->hasMany(Career::class);
    }

    public function sections()
    {
        return $this->hasMany(Section::class);
    }

    public function attendanceMethods()
    {
        return $this->hasMany(AttendanceMethod::class);
    }

    public function employees()
    {
        return $this->hasMany(Employee::class);
    }

    public function employeeTransactions()
    {
        return $this->hasMany(EmployeeTransaction::class);
    }

    public function verified(): bool
    {
        return !is_null($this->email_verified_at);
    }

    public function scopeVerified(Builder $query): void
    {
        $query->whereNotNull('email_verified_at');
    }

    public function getLogoAttribute($value)
    {
        return !is_null($value) ? asset('public/' . Company::UPLOADED_FILES . "/" . $value)
            : asset('public/' . Company::UPLOADED_FILES . "/default_image.jpg");
    }
}
