<?php

namespace App\Models;

use App\Models\Category;
use App\Http\Resources\ProductResource;
use Illuminate\Database\Eloquent\Model;
use Spatie\Translatable\HasTranslations;
use App\Http\Enums\Product\ProductTypeEnum;
use App\Http\Helper\CompanyHelper;
use App\Http\Services\Account\AccountService;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Product extends Model
{
    use HasFactory, SoftDeletes, HasTranslations;
    const UPLOADED_FILES = 'uploads/products';

    const RULES = [
        'product_details.*' => 'required',
        'product_details.name' => 'nullable|array',
        'product_details.image' => 'nullable|image',
        'product_details.name_ar' => 'nullable|string',
        'product_details.name_en' => 'nullable|string',
        'product_details.status' => 'nullable|boolean',
        'product_details.description' => 'nullable',
        'product_details.description_ar' => 'nullable|string',
        'product_details.description_en' => 'nullable|string',
        'product_details.category_id' => 'nullable|exists:categories,id',
        'product_details.has_expiry_date' => 'nullable|boolean',
        'product_details.tax_id' => 'nullable|exists:taxes,id',
         'product_details.is_stock' =>'required|in:0,1'
    ];

    protected $fillable = [
        'code',
        'company_id',
        'name',
        'barcode',
        'description',
        'category_id',
        'size',
        'has_expiry_date',
        'status',
        'tax_id',
        'image',
        'key',
        'type',
        'is_stock'
    ];

    protected $casts = [
        'type' => ProductTypeEnum::class,
    ];

    public $translatable = [
        'name',
        'description',
    ];

    public function resolveResource()
    {
        return ProductResource::class;
    }

    public function tax()
    {
        return $this->belongsTo(Tax::class);
    }

    public function category()
    {
        return $this->belongsTo(Category::class);
    }

    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    public function units()
    {
        return $this->hasMany(ProductUnit::class);
    }

    public function warehouses()
    {
        return $this->hasMany(ProductWarehouse::class);
    }
    public function branchs()
{
    return $this->hasMany(ProductBranch::class, 'product_id', 'id');
}

    public function productServiceUnits()
    {
        return $this->hasMany(ProductServiceUnit::class, 'product_id');
    }

    public function journals()
    {
        return $this->morphMany(Journal::class, 'journalable');
    }

    public function productQuantityTracks()
    {
        return $this->hasMany(ProductQuantityTrack::class);
    }

    public function dismissalNotice()
    {
        return $this->hasMany(DismissalNotice::class);
    }

    public function billJournals()
    {
        return $this->hasManyThrough(Journal::class, BillProduct::class, 'product_id', 'journalable_id')
            ->where('journalable_type', Bill::class);
    }

    public function openBalanceJournal(ProductWarehouse $product_warehouse)
    {
        return $this->journals()
            ->whereHas('entries', function ($query) use ($product_warehouse) {
                $query->where([
                    'type_type' => ProductWarehouse::class,
                    'type_id' => $product_warehouse->id,
                ]);
            });
    }

    public static function account()
    {
        return AccountService::getAccountForModel('products');
    }

    public function getImagePathAttribute($value)
    {
        return  $this->image ? asset('public/' . self::UPLOADED_FILES . '/' . $this->image) : null;
    }
}
