<?php

namespace App\Http\Controllers\Api\Auth;

use App;
use App\Models\Company;
use App\Trait\ApiTrait;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Resources\UserResource;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use App\Http\Requests\Api\Auth\LoginRequest;
use App\Http\Resources\AutoShiftResource;
use App\Http\Resources\PackageResource;
use App\Http\Resources\PermissionResource;
use App\Http\Resources\ShiftResource;
use App\Http\Services\Worker\ShiftPermissionAfterLogin;
use App\Models\AutoShift;
use App\Models\Employee;
use App\Models\Shift;
use App\Models\Worker;
use Illuminate\Validation\ValidationException;

class LoginController extends Controller
{
    use ApiTrait;

    public function login(LoginRequest $request)
    {
        $companyResponse = $this->companyLogin($request->code, $request->password);
        if ($companyResponse) {
            return $companyResponse;
        }
        $employeeResponse = $this->employeeLogin($request->code, $request->password);
        if ($employeeResponse) {
            return $employeeResponse;
        }
    }

    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();
        return $this->apiResponse(message: "Logged out successfully.");
    }

    public function loggedInUser()
    {
        $user = Company::where('id', auth()->id())->get();
        $user = UserResource::collection($user);
        return $this->apiResponse(data: $user);
    }

    private function companyLogin($code, $password)
    {
        $company = Company::where('phone', $code)->first();

        if (!$company || !Hash::check($password, $company->password)) {
            throw ValidationException::withMessages([
                'phone' => trans('auth.failed'),
            ]);
        }

        $features = $company->companyCurrentPackageWithFeatures() ?: null;

        return $this->apiResponse(
            message: __('auth.login_successfully'),
            data: [
                'id' => $company->id,
                'name' => $company->name,
                'logo' => $company->logo,
                'phone' => $company->phone,
                'email' => $company->email,
                'steps' => $company->steps,
                'is_company' => auth('company')->check(),
                'email_verified' => !is_null($company->email_verified_at),
                'token' => $company->createToken('company')->plainTextToken,
                'permissions' => PermissionResource::collection($company->permissions),
                'active_package' => $features ? PackageResource::make($features) : null,
            ]
        );
    }
    private function employeeLogin($code, $password)
    {
        $employee = Employee::where('code', $code)->first();

        if (!$employee || !Hash::check($password, $employee->password)) {
            throw ValidationException::withMessages([
                'code' => [
                    __('auth.invalid_code_or_password')
                ]
            ]);
        }

        $worker = Worker::where('employee_id', $employee->id)->firstOr(function () {
            throw ValidationException::withMessages([
                'code' => [
                    __('auth.invalid_code_or_password')
                ]
            ]);
        });

        $autoShift = AutoShift::whereHas(
            'autoShiftDetails',
            fn ($q) => $q->where('worker_id', $worker->id)
        )
            ->active()
            ->with(
                [
                    'branch',
                    'autoShiftDetails.worker',
                    'autoShiftDetails.treasury',
                    'autoShiftDetails' => fn ($q) => $q->where('worker_id', $worker->id)
                ]
            )
            ->latest()
            ->first();

        $shift = Shift::where(
            [
                'employee_id' => $employee->id,
            ]
        )
            ->whereNull('close_name')
            ->latest()
            ->first();

        //must create token here before assign permissions to worker (to compare token time and permissionsTime)
        $token = $worker->createToken('worker')->plainTextToken;
        //assign permission for open and close the shift if the worker can not open the shift and and close (edit)

        ShiftPermissionAfterLogin::addPermissionToWorker($worker, $shift, $autoShift);

        return $this->apiResponse(
            data: [
                'token' => $token,
                'shift' =>  $shift ? ShiftResource::make($shift) : null,
                'auto_shift' => ($autoShift && !$shift) ? AutoShiftResource::make($autoShift) : null,
                'permissions' => PermissionResource::collection($worker->getAllPermissions()),
            ],
            message: __('auth.login_successfully')
        );
    }
}
