<?php

namespace App\Http\Controllers\Api\Company\Journal;

use App\Models\Journal;
use App\Trait\ApiTrait;
use App\Trait\UploadFileTrait;
use App\Http\Helper\CompanyHelper;
use App\Http\Controllers\Controller;
use App\Http\Resources\JournalResource;
use App\Http\Services\Account\JournalService;
use App\Http\Requests\Api\Journal\StoreJournalRequest;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

class JournalController extends Controller
{
    use ApiTrait, UploadFileTrait;

    public function index(Request $request)
    {
        return $this->apiResponse(
            data: JournalResource::collection(
                CompanyHelper::getCompany(request())
                    ->journals()
                    ->when(
                        $request->has('report'),
                        fn ($q) => $q->with([
                            'entries.account',
                            'entries.type',
                            'journalable',
                            'employee',
                        ])
                    )
                    ->when(
                        $request->has('from'),
                        fn ($q) => $q->whereDate('created_at', '>=', $request->input('from'))
                    )
                    ->when(
                        $request->has('to'),
                        fn ($q) => $q->whereDate('created_at', '<=', $request->input('to'))
                    )
                    ->get()
            ),
        );
    }

    public function store(StoreJournalRequest $request)
    {
        $debit_side = JournalService::handleSide(
            self::filterAccountsSide(
                accounts: $request->accounts,
                side: 'debit'
            ),
            'debit'
        );

        $credit_side = JournalService::handleSide(
            self::filterAccountsSide(
                accounts: $request->accounts,
                side: 'credit'
            ),
            'credit'
        );

        if ($debit_side['side_amount'] !== $credit_side['side_amount']) {
            throw ValidationException::withMessages(['debit' => ['Debit side must be equal to credit side.']]);
        }

        $file = $this->uploadFile(Journal::UPLOADED_FILES, $request->file);

        $journal = JournalService::createJournal(
            date: $request->date ?? now(),
            type: Journal::MANUAL_TYPE,
            source: Journal::MANUAL_TYPE,
            description: $request->description,
            file: $file,
            employee: null,
            status: true,
            debit: $debit_side['side_data'],
            credit: $credit_side['side_data'],
        );

        $journal->load(['entries.account']);

        return $this->apiResponse(
            data: new JournalResource($journal),
            message: "Journal created successfully.",
        );
    }

    public function show(Journal $journal)
    {
        $journal->load([
            'entries.account',
            'entries.type',
            'journalable',
            'employee',
        ]);

        return $this->apiResponse(
            data: new JournalResource($journal),
        );
    }

    public function testing()
    {
        return $this->apiResponse(
            data: JournalResource::collection(
                CompanyHelper::getCompany(request())->journals()->with([
                    'entries.account',
                    'entries.type',
                    'journalable',
                    'employee',
                ])->get()
            ),
        );
    }

    public static function filterAccountsSide(array $accounts, string $side): array
    {
        return array_filter($accounts, fn ($account) => $account['type'] === $side);
    }
}
