<?php

namespace App\Http\Controllers\Api\Product;


use App\Models\Product;
use App\Trait\ApiTrait;
use App\Http\Helper\CompanyHelper;
use App\Http\Controllers\Controller;
use App\Http\Services\ProductService;
use App\Http\Resources\ProductResource;
use App\Http\Requests\Api\Product\StoreProductRequest;
use App\Http\Requests\Api\Product\UpdateProductRequest;

class ProductController extends Controller
{
    use ApiTrait;

    public function __construct(public ProductService $productService)
    {
    }

    public function index()
    {
        $this->authorize('view products');
        return $this->apiResponse(
            data: ProductResource::collection(
                Product::whereBelongsTo(CompanyHelper::getCompany(request()))
                    ->with([
                        'units',
                        'tax',
                        'warehouses.warehouse',
                        'category',
                    ])
                    ->get()
            )
        );
    }

    public function show(Product $product)
    {
        $this->authorize('show products');
        $product->load([
            'tax',
            'warehouses',
            'units',
        ]);
        return $this->apiResponse(data: ProductResource::make($product));
    }

    public function store(StoreProductRequest $request)
    {
        $this->authorize('create products');
        return $this->apiResponse(
            message: 'Product created successfully.',
            data: ProductResource::make($this->productService->create($request))
        );
    }

    public function update(UpdateProductRequest $request, Product $product)
    {
        $this->authorize('edit products');
        $product = $this->productService->update($request, $product);
        return $this->apiResponse(message: 'Product updated successfully.', data: ProductResource::make($product));
    }

    public function changeStatus(Product $product)
    {
        $this->authorize('edit products');
        $product->update([
            'status' => !$product->status
        ]);
        return $this->apiResponse(message: 'Product status Changed successfully.', data: ProductResource::make($product));
    }

    public function destroy(Product $product)
    {
        $this->authorize('delete products');
        $product->delete();
        return $this->apiResponse(message: 'Product deleted successfully.');
    }
    public function deleteAll()
    {
        $this->authorize('delete products');
        Product::where('company_id', CompanyHelper::getId())->delete();
        return $this->apiResponse(message: 'Product deleted successfully.');
    }
}
