<?php

namespace App\Http\Requests\Api\Bound;

use App\Models\Client;
use App\Models\Supplier;
use App\Http\Helper\CompanyHelper;
use Illuminate\Validation\Validator;
use App\Rules\ValidPaymentMethodRule;
use App\Http\Enums\Bound\BoundTypeEnum;
use App\Models\Employee;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;

class BoundRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $company = CompanyHelper::getCompany();

        return [
            'date' => 'required|date',
            'description' => 'required|string',
            // 'tax_id' => 'nullable|exists:taxes,id,company_id,' . $company->id,
            'file' => 'nullable|file',
            'payment_method_id' => [
                'required',
                'exists:payment_methods,id',
                new ValidPaymentMethodRule,
            ],
            'payment_id' => 'required|integer|min:1',
            'customer_id' => 'required|integer|min:1',
            'value' => 'required|numeric',
            'ref_number' => 'nullable|string',
            'tax_number' => 'nullable|string',
            'voucher_type' => 'required|string|in:supplier,client,employee',
        ];
    }

    protected function withValidator(Validator $validator)
    {
        $validator->after(function ($validator) {
            $this->validateCustomer($validator);
            $this->validatePaymentMethod($validator);
        });
    }

    public static function getCustomerModel()
    {
        return match (request()->voucher_type) {
            'supplier' => Supplier::class,
            'client' => Client::class,
            'employee' => Employee::class,
            default => Client::class,
        };
    }

    protected function validateCustomer(Validator $validator)
    {
        $customer_type = self::getCustomerModel();

        if (!$customer_type::where([
            'id' => $this->customer_id,
            'company_id' => CompanyHelper::getCompany(request())->id
        ])->exists()) {
            throw ValidationException::withMessages([
                'customer_id' => ['Invalid customer.']
            ]);
        }

        $this->merge([
            'entry_type' => $customer_type,
            'entry_id' => $this->customer_id,
            'company_id' => CompanyHelper::getCompany(request())->id,
            'type' => BoundTypeEnum::getValueFromString(request()->route()->type)
        ]);
    }

    protected function validatePaymentMethod(Validator $validator)
    {
        $paymentMethod = CompanyHelper::getCompany(request())
            ->paymentMethods()
            ->where('payment_methods.id', $this->payment_method_id ?? null)
            ->first();

        if (!$paymentMethod) {
            throw ValidationException::withMessages([
                'payment_method_id' => ['Invalid payment method.']
            ]);
        }

        $this->merge([
            'payment_method_id' => $paymentMethod->id,
            'payment_type' => $paymentMethod->model,
            'resource' => $paymentMethod->resource,
            'payment_id' => $this->payment_id,
        ]);
    }
}
