<?php

namespace App\Http\Services\DismissalNotice;

use App\Models\Journal;
use App\Models\Product;
use App\Models\DismissalNotice;
use App\Models\ProductWarehouse;
use App\Http\Helper\AccountHelper;
use Illuminate\Support\Facades\DB;
use App\Http\Services\Account\AccountService;
use App\Http\Services\Account\JournalService;
use Illuminate\Validation\ValidationException;

class DismissalNoticeService
{
    const NAME = Journal::AUTO_TYPE;
    const DESCRIPTION = "Dismissal Notice Description";

    public static function createDismissal(array $data): DismissalNotice
    {
        try {
            DB::beginTransaction();
            $productWarehouse = self::productWarehouse(
                productId: $data['product_id'],
                warehouseId: $data['warehouse_id']
            );

            if (!self::quantityAvailable(productWarehouse: $productWarehouse, quantity: $data['quantity'])) {
                throw ValidationException::withMessages([
                    'quantity' => ['Quantity not enough for this product in warehouse.']
                ]);
            }

            $dismissalNotice = DismissalNotice::create(['use_quantity' => $data['quantity']] + $data);

            self::createDismissalNoticeJournal(
                amount: $productWarehouse->price *  $data['quantity'],
                product: $productWarehouse->product,
                dismissalNotice: $dismissalNotice,
            );

            self::updateWarehouseQuantity(productWarehouse: $productWarehouse, quantity: $data['quantity']);

            DB::commit();

            return $dismissalNotice;
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    public static function createDismissalNoticeJournal($amount, Product $product, DismissalNotice $dismissalNotice)
    {
        JournalService::createJournal(
            date: now(),
            type: self::NAME,
            source: 'Dismissal Notice',
            description: self::DESCRIPTION,
            file: null,
            employee: null,
            status: true,
            debit: [
                [
                    'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_SUPPLIES_WAREHOUSE),
                    'type' => [
                        'type' => DismissalNotice::class,
                        'id' => $dismissalNotice->id,
                    ],
                    'amount' => $amount,
                    'entry_type' => "debit",
                ],
            ],
            credit: [
                [
                    'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_PRODUCTS),
                    'type' => [
                        'type' => Product::class,
                        'id' => $product->id,
                    ],
                    'amount' => $amount,
                    'entry_type' => "credit",
                ],
            ],
            journalable: $product
        );
    }

    public static function productWarehouse($productId, $warehouseId): ProductWarehouse
    {
        return ProductWarehouse::where([
            'warehouse_id' => $warehouseId,
            'product_id' => $productId,
        ])->with(['product:id'])
            ->firstOr(['quantity', 'value', 'price', 'product_id'], function () {
                throw ValidationException::withMessages([
                    'warehouse_id' => ['product not in warehouse.']
                ]);
            });
    }

    public static function quantityAvailable(ProductWarehouse $productWarehouse, $quantity): bool
    {
        return $productWarehouse->quantity >= $quantity;
    }

    public static function updateWarehouseQuantity(ProductWarehouse $productWarehouse, $quantity): void
    {
        $warehouse_new_quantity = $productWarehouse->quantity - $quantity;

        $productWarehouse->update([
            'quantity' => $warehouse_new_quantity,
            'value' => $warehouse_new_quantity * $productWarehouse->price,
        ]);
    }
}
