<?php

namespace App\Http\Services\Online\V1\Order;

use App\Http\Helper\CompanyHelper;
use App\Models\Order;
use Illuminate\Support\Str;
use App\Http\Requests\Online\V1\OrderRequest;
use Illuminate\Support\Facades\DB;

class OrderService
{
    const PREFIX = 'ORDER:';

    private Order $order;

    public function createOnlineOrder(OrderRequest $request) : string
    {
        DB::transaction(function () use ($request) {
            // create main order
            $this->createOrder(
                bill_type: $request->bill_type,
                note: $request->note,
                total: $request->transaction['amount']['total'],
                paid_amount: $request->transaction['amount']['paid_amount'],
                subtotal: $request->transaction['amount']['subtotal'],
                tax: $request->transaction['amount']['tax'],
                discount: $request->transaction['amount']['discount'],
                user_id: $request->transaction['sub_details']['user_id'],
                cart_id: $request->transaction['sub_details']['cart_id'],
            )
                ->createOrderUser(
                    name: $request->transaction['customer_details']['name'],
                    email: $request->transaction['customer_details']['email'],
                    phone: $request->transaction['customer_details']['phone'],
                    country: $request->transaction['customer_details']['country'],
                    city: $request->transaction['customer_details']['city'],
                    address: $request->transaction['customer_details']['address'],
                )
                ->createItems($request->transaction['item_list']);
        });

        return $this->order->ref_id;
    }

    private function createOrder(
        string $bill_type = 'sales',
        ?string $note = null,
        ?int $total = null,
        ?int $paid_amount = null,
        ?int $subtotal = null,
        ?int $tax = null,
        ?int $discount = null,
        ?int $user_id = null,
        ?int $cart_id = null,
    ): self {

        $this->order = CompanyHelper::getCompany()->orders()->create([
            'ref_id' => self::createRefCode(),
            'bill_type' => $bill_type,
            'note' => $note,
            'total' => $total,
            'paid_amount' => $paid_amount,
            'subtotal' => $subtotal,
            'tax' => $tax,
            'discount' => $discount,
            'user_id' => $user_id,
            'cart_id' => $cart_id,
        ]);

        return $this;
    }

    private function createOrderUser(
        ?string $name,
        ?string $email,
        ?string $phone,
        ?string $country,
        ?string $city,
        ?string $address,
    ): self {

        $this->order->user()->create([
            'name' => $name,
            'email' => $email,
            'phone' => $phone,
            'country' => $country,
            'city' => $city,
            'address' => $address,
        ]);

        return $this;
    }

    private function createItems(array $items): self
    {

        foreach ($items as $item) {
            $this->createItem(
                key: $item['key'],
                name: $item['name'],
                quantity: $item['quantity'],
                price: $item['price'],
                tax: $item['tax'],
                discount: $item['discount'],
            );
        }

        return $this;
    }

    private function createItem(
        string $key,
        string $name,
        int $quantity,
        float|int $price,
        float|int $tax,
        float|int $discount,
    ): self {

        $this->order->items()->create([
            'key' => $key,
            'name' => $name,
            'quantity' => $quantity,
            'price' => $price,
            'tax' => $tax,
            'discount' => $discount,
        ]);

        return $this;
    }

    private static function createRefCode(): string
    {
        $keyExists = true;
        $key = '';

        while ($keyExists) {
            $randomKey = Str::random(15);
            $key = self::PREFIX . $randomKey;

            $keyExists = Order::where('ref_id', $key)->exists();
        }

        return $key;
    }
}
