<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Spatie\Translatable\HasTranslations;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Account extends Model
{
    use HasFactory, HasTranslations, SoftDeletes;

    const ROLE = [
        'name' => 'nullable|array',
        'description' => 'nullable|array',
        'name_ar' => 'required|string',
        'name_en' => 'required|string',
        'description_ar' => 'required|string',
        'description_en' => 'required|string',
        'active' => 'required|boolean',
    ];

    protected $fillable = [
        'name',
        'description',
        'code',
        'parent_id',
        'type', // 'debit -credit'
        'status', // 'main -sub'
        'company_id',
        'active',
        'amount',
        'level_id',
    ];

    public $translatable = [
        'name',
        'description',
    ];

    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    public function parent()
    {
        return $this->belongsTo(Account::class, 'parent_id')->with('parent');
    }

    public function level()
    {
        return $this->belongsTo(Level::class);
    }

    public function children()
    {
        return $this->hasMany(Account::class, 'parent_id')->with('children');
    }

    public function entries()
    {
        return $this->hasMany(JournalEntry::class);
    }

    public function childrenWithEntries()
    {
        return $this->children()->with('entries', 'children.entries');
    }

    public function model()
    {
        return $this->belongsToMany(MainModel::class, 'account_model', 'account_id', 'model_id');
    }

    public function setAmountAttribute($value)
    {
        $this->attributes['amount'] = $value * 100;
    }

    public function getAmountAttribute($value)
    {
        return $value / 100;
    }

    public function scopeCodeSearch($query, $code)
    {
        $query->where('code', 'LIKE', "%" . "$code" . "%");
    }
}
