<?php

namespace App\Models;

use App\Models\Category;
use App\Http\Resources\ProductResource;
use Illuminate\Database\Eloquent\Model;
use Spatie\Translatable\HasTranslations;
use App\Http\Enums\Product\ProductTypeEnum;
use App\Http\Services\Account\AccountService;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Http\Services\Account\OpenBalanceService;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Product extends Model
{
    use HasFactory, SoftDeletes, HasTranslations;
    const RULES = [
        'product_details.*' => 'required',
        'product_details.key' => 'nullable',
        'product_details.name' => 'nullable',
        'product_details.name_ar' => 'required|string',
        'product_details.name_en' => 'required|string',
        'product_details.barcode' => 'required|string',
        'product_details.description' => 'nullable',
        'product_details.description_ar' => 'required|string',
        'product_details.description_en' => 'required|string',
        'product_details.category_id' => 'required|exists:categories,id',
        // 'product_details.size' => 'required|numeric',
        'product_details.inventory_type' => 'required|between:1,4',
        'product_details.has_expiry_date' => 'boolean',
        'product_details.tax_id' => 'required|exists:taxes,id',

    ];

    const INVENTORY_DAILY = 1;
    const INVENTORY_WEEKLY = 2;
    const INVENTORY_MONTHLY = 3;
    const INVENTORY_ANNUAL = 4;

    protected $fillable = [
        'code',
        'name',
        'barcode',
        'description',
        'category_id',
        'size',
        'inventory_type',
        'has_expiry_date',
        'status',
        'tax_id',
        'key',
        'type',
    ];

    protected $casts = [
        'type' => ProductTypeEnum::class,
    ];

    public $translatable = [
        'name',
        'description',
    ];

    public function resolveResource()
    {
        return ProductResource::class;
    }

    public function tax()
    {
        return $this->belongsTo(Tax::class);
    }

    public function category()
    {
        return $this->belongsTo(Category::class);
    }

    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    public function units()
    {
        return $this->hasMany(ProductUnit::class);
    }

    public function warehouses()
    {
        return $this->hasMany(ProductWarehouse::class);
    }

    public function productServiceUnits()
    {
        return $this->hasMany(ProductServiceUnit::class, 'product_id');
    }

    public function journals()
    {
        return $this->morphMany(Journal::class, 'journalable');
    }

    public function dismissalNotice()
    {
        return $this->hasMany(DismissalNotice::class);
    }

    public function billJournals()
    {
        return $this->hasManyThrough(Journal::class, BillProduct::class, 'product_id', 'journalable_id')
            ->where('journalable_type', Bill::class);
    }

    public function openBalanceJournal(ProductWarehouse $product_warehouse)
    {
        return $this->journals()
            ->where(['type' => OpenBalanceService::NAME])
            ->whereHas('entries', function ($query) use ($product_warehouse) {
                $query->where([
                    'type_type' => ProductWarehouse::class,
                    'type_id' => $product_warehouse->id,
                ]);
            });
    }

    public static function account()
    {
        return AccountService::getAccountForModel('products');
    }
}
