<?php

namespace App\Trait;

use App\Models\Tax;
use App\Models\Bill;
use App\Models\Client;
use App\Models\Supplier;
use App\Models\Warehouse;
use App\Models\BillProduct;
use App\Models\BillService;
use App\Models\ProductUnit;
use App\Models\ProductWarehouse;
use App\Http\Enums\Bill\BillEnum;
use App\Http\Services\Bill\BillProductService;
use App\Http\Services\Bill\BillServiceService;

trait BillTrait
{
    private function calcDiscount(float $price = 0, string $discount_type, float $discount_amount): float
    {
        if ($discount_type == "percentage") {
            $discount  = $price * ($discount_amount / 100);
            $this->discount += $discount;
            return $discount;
        }

        $this->discount += $discount_amount;
        return $discount_amount;
    }

    private function calcRequestedProductValue(int $quantity, float $price): float
    {
        $total = $quantity * $price;
        $this->total += $total;
        return $total;
    }

    private function tax(int $taxId): Tax
    {
        return Tax::findOrFail($taxId);
    }

    private function calcTaxValue(Tax $tax, float $price, float $discount): float
    {
        $tax = ($price - $discount) * ($tax->rate / 100);
        $this->tax += $tax;
        return $tax;
    }

    private function calculateRemainingAmount(float $paid_amount): float
    {
        dd($this->discount);
        return (($this->total - $this->discount) + $this->tax) -  $paid_amount;
    }

    private function calcQuantityByMainUnit(ProductUnit $productUnit, int $quantity): float
    {
        return $productUnit->conversion_factor * $quantity;
    }

    private function prepareValues(array $data, mixed $key): array
    {
        return array_diff_key($data, [$key => 1, 'id' => 2]);
    }

    private function isSalesBill(): bool
    {
        return $this->bill->type === BillEnum::sales->value;
    }

    public static function getBillableClass(string $route_type)
    {
        return $route_type == BillEnum::sales->value ? Client::class : Supplier::class;
    }

    public static function calcTaxOnBill(float $tax_percentage, float $bill_total): float
    {
        return $bill_total * $tax_percentage;
    }

    public static function calcTaxOnProduct(float $tax_percentage, float $product_price): float
    {
        return $product_price * $tax_percentage;
    }

    public function setBill(Bill $bill): self
    {
        $this->bill = $bill;
        return $this;
    }

    private function calculateUpdatedProductValues(int $currentQuantity, int $newQuantityAmount, float $price, ProductWarehouse $productWarehouse): array
    {
        if ($this->isSalesBill()) {
            return $this->calculateValuesForSalesBill($currentQuantity, $newQuantityAmount, $price, $productWarehouse);
        }

        return $this->calculateValuesForNonSalesBill($currentQuantity, $newQuantityAmount, $price, $productWarehouse);
    }

    private function calculateValuesForSalesBill(int $currentQuantity, int $newQuantityAmount, ?float $price, ProductWarehouse $productWarehouse): array
    {
        $newQuantity = $currentQuantity - $newQuantityAmount;

        return [
            'value'    => $newQuantity * $productWarehouse->price,
            'quantity' => $newQuantity,
            'price'    => $productWarehouse->price,
        ];
    }

    private function calculateValuesForNonSalesBill(int $currentQuantity, int $newQuantityAmount, float $price, ProductWarehouse $productWarehouse): array
    {
        $newValue = $productWarehouse->value + ($newQuantityAmount * $price);
        $newQuantity = $currentQuantity + $newQuantityAmount;

        return [
            'value'    => $newValue,
            'quantity' => $newQuantity,
            'price'    => $newValue / $newQuantity,
        ];
    }

    private function getBillProductsIds(): array
    {
        return BillProduct::where('bill_id', $this->bill->id)->pluck('product_id')->toArray();
    }

    private function getBillServicesIds(): array
    {
        return BillService::where('bill_id', $this->bill->id)->pluck('service_id')->toArray();
    }

    private function deleteUnwantedProducts(array $productIds): void
    {
        BillProduct::where('bill_id', $this->bill->id)->whereIn('product_id', $productIds)->delete();
    }

    private function deleteUnwantedServices(array $serviceIds): void
    {
        BillService::where('bill_id', $this->bill->id)->whereIn('service_id', $serviceIds)->delete();
    }

    private function updateProperties(): void
    {
        $billProductServiceTotal = isset($this->billProductService) ? $this->billProductService->total : 0;
        $billServiceServiceTotal = isset($this->billServiceService) ? $this->billServiceService->total : 0;
        $billProductServiceCost = isset($this->billProductService) ? $this->billProductService->cost : 0;
        $billServiceServiceCost = isset($this->billServiceService) ? $this->billServiceService->cost : 0;
        $billProductServiceDiscount = isset($this->billProductService) ? $this->billProductService->discount : 0;
        $billServiceServiceDiscount = isset($this->billServiceService) ? $this->billServiceService->discount : 0;
        $billProductServiceTax = isset($this->billProductService) ? $this->billProductService->tax : 0;
        $billServiceServiceTax = isset($this->billServiceService) ? $this->billServiceService->tax : 0;

        $this->total = $billProductServiceTotal + $billServiceServiceTotal;
        $this->productsCost = $billProductServiceCost;
        $this->servicesCost = $billServiceServiceCost;
        $this->discount = $billProductServiceDiscount + $billServiceServiceDiscount;
        $this->tax = $billProductServiceTax + $billServiceServiceTax;
    }

    public static function requestHave(string $value, $request): bool
    {
        return isset($request[$value]);
    }

    private function updateProductQuantitiesAndDismissalNotices(): void
    {
        if ($this->billWithProducts) {
            $this->billProductService->updateProductsQuantities();
        }

        if ($this->billWithServices && $this->bill->isSalesBill()) {
            $this->billServiceService->updateDismissalNotices();
        }
    }

    private function initializeBillProductService(array $billProductsData, Warehouse $warehouse): void
    {
        $this->billProductService = new BillProductService(
            billProductsData: $billProductsData,
            warehouse: $warehouse
        );

        $this->billProductService->processRequestedProducts();
    }

    private function initializeBillServiceService(array $billServicesData): void
    {
        $this->billServiceService = new BillServiceService(
            billServicesData: $billServicesData,
            warehouse: $this->warehouse
        );

        $this->billServiceService->processRequestedServices();
    }

    private static function getWarehouse(int $warehouseId): Warehouse
    {
        return Warehouse::where('id', $warehouseId)->firstOrFail();
    }
}
