<?php

namespace App\Http\Controllers\Api\Auth;


use App\Models\Company;
use App\Models\UserOtp;
use App\Trait\ApiTrait;
use App\Mail\PasswordChanged;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use App\Jobs\SendOtpEmailForgetPassword;
use Illuminate\Validation\ValidationException;
use App\Http\Requests\Api\Company\VerifyEmailRequest;
use App\Http\Requests\Api\Company\ForgetPassword\CheckOtpRequest;
use App\Http\Requests\Api\Company\ForgetPassword\ResetPasswordRequest;
use App\Http\Requests\Api\Company\ForgetPassword\ForgetPasswordRequest;

class ForgetPasswordController extends Controller
{
    use ApiTrait;

    public function sendOtp(ForgetPasswordRequest $request)
    {
        Company::where('email', $request->email)->firstOr(function () {
            throw ValidationException::withMessages([
                'email' => 'Email does not exists.',
            ]);
        });

        UserOtp::where('email', $request->email)->delete();

        SendOtpEmailForgetPassword::dispatch($request->email);

        return $this->apiResponse(message: __('messages.forget_password_code_sent_successfully'));
    }

    public function checkOtpCode(CheckOtpRequest $request)
    {
        self::validateCode($request->code);

        return $this->apiResponse(message: __('messages.code_is_valid'));
    }

    public function resetPassword(ResetPasswordRequest $request)
    {
        $otp = self::validateCode($request->code);

        $company = Company::where('email', $request->email)->firstOr(function () {
            throw ValidationException::withMessages([
                'email' => __('messages.email_does_not_exist'),
            ]);
        });

        if ($otp->email != $company->email) {
            throw ValidationException::withMessages([
                'email' => __('messages.email_does_not_match'),
            ]);
        }

        $company->update(['password' => Hash::make($request->password)]);

        $otp->delete();

        $company->tokens()->delete();

        Mail::to($company->email)->send(new PasswordChanged());

        return $this->apiResponse(message: __('messages.password_reset_successfully'));
    }

    private static function validateCode(string $code): UserOtp
    {
        $otp = UserOtp::where('code', '=', $code)->first();

        if (!$otp) {
            throw ValidationException::withMessages([
                'otp' => __('messages.invalid_code'),
            ]);
        }

        if (now() > $otp->expire_at) {
            throw ValidationException::withMessages([
                'otp' => __('messages.code_expired'),
            ]);
        }

        return $otp;
    }
}
