<?php

namespace App\Http\Controllers\Api\Branch\Warehouse;

use App\Trait\ApiTrait;
use App\Models\Warehouse;
use App\Http\Controllers\Controller;
use App\Http\Helper\CompanyHelper;
use App\Http\Resources\WarehouseResource;
use App\Http\Requests\Api\Warehouse\StoreWarehouseRequest;
use App\Http\Requests\Api\Warehouse\UpdateWarehouseRequest;
use App\Http\Requests\UpdateProductInventory;
use App\Http\Requests\UpdateProductsActualQuantity;
use App\Http\Services\ProductInventoryService;

class WarehouseController extends Controller
{
    use ApiTrait;

    public function index()
    {
        $this->authorize('view warehouses');

        return $this->apiResponse(data: WarehouseResource::collection(
            CompanyHelper::getCompany()
                ->warehouses()
                ->with(['branches',])
                ->withSum('products', 'value')
                ->get()
        ));
    }

    public function show(Warehouse $warehouse)
    {
        $this->authorize('show warehouses');

        $warehouse->load([
             'products.product.units',
           'products.warehouseUnits.unit',
            'products.product.units.lastSalesPrice',
            'products.product.units.lastPurchasePrice',
            'products.product.tax',
            'branches',
        ]);

        $warehouse->loadSum('products', 'value');

        return $this->apiResponse(data: new WarehouseResource($warehouse));
    }

    public function store(StoreWarehouseRequest $request)
    {
        $this->authorize('create warehouses');

        $warehouse = CompanyHelper::getCompany(request())->warehouses()->create($request->validated());
        $warehouse->branches()->sync($request->branch_ids);
        return $this->apiResponse(data: new WarehouseResource($warehouse), message: __('messages.warehouse_created_successfully'));
    }

    public function update(UpdateWarehouseRequest $request, Warehouse $warehouse)
    {
        $this->authorize('edit warehouses');

        $warehouse->update($request->validated());
        $warehouse->branches()->sync($request->branch_ids);
        return $this->apiResponse(data: new WarehouseResource($warehouse), message: __('messages.warehouse_updated_successfully'));
    }

    public function destroy(Warehouse $warehouse)
    {
        $this->authorize('delete warehouses');

        $warehouse->delete();
        return $this->apiResponse(message: __('messages.warehouse_deleted_successfully'));
    }
    public function updateProductsActualQuantity(UpdateProductsActualQuantity $request, Warehouse $warehouse)
    {
        foreach ($request->products as $product) {
            $warehouse->products()->where('product_id', $product['id'])->update(
                [
                    'actual_quantity' => $product['actual_quantity'],
                ]
            );
        }
        return $this->apiResponse(message: __('messages.products_actual_quantity_updated_successfully'));
    }

    public function updateProductInventory(UpdateProductInventory $request, Warehouse $warehouse)
    {
        ProductInventoryService::update($request->products, $warehouse);
        return $this->apiResponse(message: __('messages.products_settlements_updated_successfully'));
    }
}
