<?php

namespace App\Http\Controllers\Api\Client;

use App\Models\Client;
use App\Models\Address;
use App\Trait\ApiTrait;
use Illuminate\Http\Request;
use App\Trait\UploadFileTrait;
use App\Http\Helper\CompanyHelper;
use App\Http\Controllers\Controller;
use App\Http\Resources\ClientResource;
use App\Http\Requests\Api\Client\AddImageRequest;
use App\Http\Requests\Api\Client\StoreClientRequest;
use App\Http\Requests\Api\Client\UpdateClientRequest;
use App\Http\Requests\FileUploadRequest;
use App\Http\Services\StoreFromExcelService;

class ClientController extends Controller
{
    use UploadFileTrait, ApiTrait;


    public function index(Request $request)
    {
        $this->authorize('view clients');

        return $this->apiResponse(data: ClientResource::collection(
            Client::whereBelongsTo(CompanyHelper::getCompany(request()))
                ->doesntHave('blocked')
                ->with(['entries' , 'bills.billPaymentMethods'])
                ->get()
        ));
    }

    public function show(Client $client)
    {
        $this->authorize('show clients');

        $client->load([
            'address.area',
            'favoriteEmployee',
            'title',
            'group',
            'address',
            'entries.journal',

        ]);
        return $this->apiResponse(data: new ClientResource($client));
    }

    public function store(StoreClientRequest $request)
    {
        $this->authorize('create clients');

        return $this->apiResponse(
            data: new ClientResource(
                Client::create([
                    'address_id' => Address::create($request->only(['area_id', 'post_code', 'address_line']))->id
                ]
                    + $request->validated())
            ),
            message: __('messages.client_created_successfully')
        );
    }

    public function storeFromExcel(FileUploadRequest $request)
    {
        $this->authorize('create clients');
        StoreFromExcelService::storeClients($request);
        return $this->apiResponse(
            message: __('messages.client_created_successfully')
        );
    }


    public function update(UpdateClientRequest $request, Client $client)
    {
        $this->authorize('edit clients');

        $client->address->update($request->only(['area_id', 'post_code', 'address_line']));
        $client->update($request->validated());
        return $this->apiResponse(data: new ClientResource($client), message: __('messages.client_updated_successfully'));
    }

    public function addImage(AddImageRequest $request, Client $client)
    {
        $this->authorize('edit clients');

        $image = $this->uploadFile(Client::UPLOAD_PATH, $request->image);
        $client = $client->update(['image' => $image]);
        return $this->apiResponse(data: new ClientResource($client), message: __('messages.client_image_added_successfully'));
    }

    public function changeStatus(Client $client)
    {
        $this->authorize('edit clients');

        $client->update([
            'status' => !$client->status->value,
        ]);

        return $this->apiResponse(message: __('messages.client_status_changed_successfully'));
    }

    public function destroy(Client $client)
    {
        $this->authorize('delete clients');

        Address::findOrFail($client->address->id)->delete();
        $client->forceDelete();
        return $this->apiResponse(message: __('messages.client_deleted_successfully'));
    }


}
