<?php

namespace App\Http\Controllers\Api\Employee;

use App\Models\Address;
use App\Trait\ApiTrait;
use App\Models\Employee;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Helper\CompanyHelper;
use App\Http\Resources\EmployeeResource;
use App\Http\Requests\Api\Employee\Employee\StoreEmployeeRequest;
use App\Http\Requests\Api\Employee\Employee\UpdateEmployeeRequest;
use Illuminate\Support\Facades\Hash;

class EmployeeController extends Controller
{
    use ApiTrait;

    public function index(Request $request)
    {
        $this->authorize('view employees');

        return $this->apiResponse(data: EmployeeResource::collection(CompanyHelper::getCompany()->employees));
    }

    public function show(Employee $employee)
    {
        $this->authorize('show employees');

        $employee->load([
            'address.area.city.country',
            'section',
            'career',
            'attendanceMethod',
            'branches'
        ]);

        return $this->apiResponse(data: EmployeeResource::make($employee));
    }

    public function store(StoreEmployeeRequest $request)
    {
        $this->authorize('create employees');

        $address = Address::create($request->only([
            'address_line',
            'area_id',
            'post_code',
        ]));
        $employee =  CompanyHelper::getCompany()
            ->employees()
            ->create([
                'address_id' => $address->id,
                'password' => Hash::make(Employee::PASSWORD)
            ] + $request->validated());

        $employee->branches()->attach($request->input('branches'));
        return $this->apiResponse(
            message: __('messages.employee_created_successfully'),
            data: EmployeeResource::make(
                $employee
            )
        );
    }

    public function update(UpdateEmployeeRequest $request, Employee $employee)
    {
        $this->authorize('edit employees');
        $validatedData = $request->validated();
        if($request->has('password')){
            $validatedData['password'] = Hash::make($request->password);
        }

        $employee->update($validatedData );
        $employee->branches()->sync($request->input('branches'));
        $employee->address()->update($request->only([
            'address_line'
        ]));
        $employee->load([
            'address',
            'section',
            'career',
            'attendanceMethod',
            'branches'
        ]);
        $employee = new EmployeeResource($employee);
        return $this->apiResponse(message: __('messages.employee_updated_successfully'), data: $employee);;
    }

    public function destroy(Employee $employee)
    {
        $this->authorize('delete employees');

        $employee->forceDelete();
        return $this->apiResponse(message: __('messages.employee_deleted_successfully'));;
    }
}
