<?php

namespace App\Http\Requests\Api\Bound;

use App\Models\Client;
use App\Models\Supplier;
use App\Http\Helper\CompanyHelper;
use Illuminate\Validation\Validator;
use App\Rules\ValidPaymentMethodRule;
use App\Http\Enums\Bound\BoundTypeEnum;
use App\Models\Account;
use App\Models\Employee;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\ValidationException;

class BoundRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        return [
            'date' => 'required|date',
            'description' => 'nullable|string',
            'payment_method_id' => [
                'required',
                'exists:payment_methods,id',
                new ValidPaymentMethodRule,
            ],

            'ref_number' => 'nullable|string',
            'branch_id' => 'required|exists:branches,id',
            'payment_id' => 'required|integer|min:1',
            'voucher_type' => 'required|string|in:supplier,client,employee,account',

            'status' => 'required|in:draft,saved,accredited',
            'entries' => 'required|array|min:1',
            'entries.*.value' => 'required|numeric',
            'entries.*.customer_id' => 'required|numeric',
            'entries.*.note' => 'nullable|string',

        ];
    }

    protected function withValidator(Validator $validator)
    {
        $validator->after(function ($validator) {
            $this->validateCustomer($validator);
            $this->validatePaymentMethod($validator);
        });
    }

    public static function getCustomerModel()
    {
        return match (request()->voucher_type) {
            'supplier' => Supplier::class,
            'client' => Client::class,
            'employee' => Employee::class,
            'account' => Account::class,
            default => Client::class,
        };
    }

    protected function validateCustomer(Validator $validator)
    {
        $customer_type = self::getCustomerModel();
        $company_id = CompanyHelper::getId();

        $entries = $this->entries;
        foreach ($this->entries as $index => $entry) {
            if (!$customer_type::where(['id' => $entry['customer_id'], 'company_id' => $company_id])->exists()) {
                $validator->errors()->add('entries.' . $index . '.customer_id', 'Invalid customer.');
            } else {
                $entries[$index]['entry_type'] = $customer_type;
                $entries[$index]['entry_id'] = $entry['customer_id'];
                $entries[$index]['value'] = $entry['value'];
                $entries[$index]['note'] = $entry['note'];
                unset($entries[$index]['customer_id']);
            }
        }

        $this->merge([
            'type' => BoundTypeEnum::getValueFromString(request()->route()->type),
            'company_id' => $company_id,
            'entries' => $entries,
        ]);
    }

    protected function validatePaymentMethod(Validator $validator)
    {
        $paymentMethod = CompanyHelper::getCompany(request())
            ->paymentMethods()
            ->where('payment_methods.id', $this->payment_method_id ?? null)
            ->first();

        if (!$paymentMethod) {
            throw ValidationException::withMessages([
                'payment_method_id' => ['Invalid payment method.']
            ]);
        }

        $this->merge([
            'payment_method_id' => $paymentMethod->id,
            'payment_type' => $paymentMethod->model,
            'resource' => $paymentMethod->resource,
            'payment_id' => $this->payment_id,
        ]);
    }
}
