<?php

namespace App\Http\Requests\Api\Journal;

use App\Http\Helper\CompanyHelper;
use App\Rules\ValidRefIdForType;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class StoreJournalRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     *
     * @return bool
     */
    public function authorize()
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array
     */
    public function rules()
    {
        $company = CompanyHelper::getCompany(request());
        $rules = [
            'reference_code' => [
                'nullable',
                'string',
                Rule::unique('journals')->where(fn($q) => $q->where('company_id', $company->id))
            ],

            'branch_id' => 'nullable|exists:branches,id',
            'warehouse_id' => 'nullable|exists:warehouses,id',
            'status' => 'required|in:draft,saved,accredited',
            'description' => 'required|string',
            'date' => 'required|date',
            'file' => 'nullable|file',
            
            'files' => ['nullable', 'array'],
            'files.*.file' => ['required_with:files', 'file'],
            'files.*.reference_number' => ['nullable', 'string', 'unique:journal_files,reference_number'],

            'accounts' => 'required|array',
            'accounts.*.amount' => 'required|numeric|min:0.01',
            'accounts.*.account' => 'required|exists:accounts,id,company_id,' . $company->id,
            'accounts.*.type' => 'required|in:credit,debit',
            'accounts.*.note' => 'nullable|string',
            'accounts.*.order' => 'required|integer',
            //for Journal reference
            'accounts.*.ref' => 'sometimes|required|array',
            'accounts.*.ref.id' => [
                'required_with:accounts.*.ref',
                function ($attribute, $value, $fail) use ($company) {
                    preg_match('/accounts\.(\d+)\.ref\.id/', $attribute, $matches);
                    $index = $matches[1] ?? null;
                    $type = $this->input("accounts.$index.ref.type");
                    if ($type && !(new ValidRefIdForType($type))->passes($attribute, $value)) {
                        $fail("The reference ID is invalid for the type '$type'.");
                    }
                },
            ],
            'accounts.*.ref.type' => 'required_with:accounts.*.ref|in:client,supplier',
           
            // for Journal Repeation Notifications handling


        ];
        if ($this->has('repeatable') && $this->input('repeatable') === true) {
            $rules['period'] = 'required|integer|in:1,2,3,4,5,6';
            $rules['start_date'] = 'required|date';
            $rules['end_date'] = 'required|date|after:start_date';
            $rules['desc'] = 'nullable|string';
        }
        return $rules;
    }
}
