<?php

namespace App\Http\Requests\Api\Journal;

use App\Http\Helper\CompanyHelper;
use App\Rules\ValidRefIdForType;
use Illuminate\Foundation\Http\FormRequest;

class UpdateJournalRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {

        $company = CompanyHelper::getCompany(request());
        $rules = [
            'date' => 'nullable|date',
            'branch_id' => 'nullable|exists:branches,id',
            'description' => 'nullable|string',
            'file' => 'nullable|file',
            'status' => 'required|in:draft,saved,accredited',
            'accounts' => 'required|array',
            'accounts.*.amount' => 'required|numeric|min:0.01',
            'accounts.*.account' => 'required|exists:accounts,id,company_id,' . CompanyHelper::getId(),
            'accounts.*.type' => 'required|in:credit,debit',
            'accounts.*.note' => 'nullable|string',
            'accounts.*.ref' => 'sometimes|required|array',
            'accounts.*.order' => 'required|integer',
            'accounts.*.ref.id' => [
                'required_with:accounts.*.ref',
                function ($attribute, $value, $fail) use ($company) {
                    preg_match('/accounts\.(\d+)\.ref\.id/', $attribute, $matches);
                    $index = $matches[1] ?? null;
                    $type = $this->input("accounts.$index.ref.type");
                    if ($type && !(new ValidRefIdForType($type))->passes($attribute, $value)) {
                        $fail("The reference ID is invalid for the type '$type'.");
                    }
                },
            ],
            'accounts.*.ref.type' => 'required_with:accounts.*.ref|in:client,supplier',
        ];

        if ($this->has('repeatable') && $this->input('repeatable') === true) {
            $rules['period'] = 'required|integer|in:1,2,3,4,5';
            $rules['start_date'] = 'required|date';
            $rules['end_date'] = 'required|date|after:start_date';
            $rules['desc'] = 'nullable|string';
        }
        return $rules;
    }
}
