<?php

namespace App\Http\Requests\Api\Product;

use App\Models\Product;
use App\Models\ProductUnit;
use Illuminate\Validation\Rule;
use App\Models\ProductWarehouse;
use App\Http\Helper\CompanyHelper;
use Illuminate\Database\Query\Builder;
use App\Http\Enums\Product\ProductTypeEnum;
use App\Models\ProductBranch;
use App\Rules\UniqueBarcodePerCompany;
use Illuminate\Foundation\Http\FormRequest;

class StoreProductRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array|string>
     */
    public function rules(): array
    {

        return array_merge(
            array_merge(Product::RULES, [
                'product_details.key' => [
                    'nullable',
                    Rule::unique('products')->where(
                        fn(Builder $query) => $query->where('company_id', CompanyHelper::getId())
                    ),
                ],
                'product_details.code' => [
                    'required',
                    Rule::unique('products')->where(function (Builder $query) {
                        return $query->where('company_id', CompanyHelper::getId());
                    }),
                ],
                'product_details.type' => 'nullable|in:' . implode(',', ProductTypeEnum::getTypes()),
            ]),
            ProductUnit::RULES,


            array_merge(
                ProductUnit::RULES,
                [
                    "product_unit.*.barcode" => [
                        'nullable',
                        'string',
                        new UniqueBarcodePerCompany(CompanyHelper::getId())
                    ],
                ]
            ),

            ProductWarehouse::RULES,
            ProductBranch::RULES
        );
    }

    public function prepareForValidation()
    {
        $last_code =  Product::where('company_id', CompanyHelper::getId())->max('code') ?? 0;

        $this->merge([
            'product_details' => array_merge($this->input('product_details') ?? [], [
               'code' => isset($this->product_details['code']) ? $this->product_details['code'] : ($last_code + 1),
                'has_expiry_date' => ($this->product_details['has_expiry_date'] ??  'off') == 'on',
                'name' => [
                    'ar' => isset($this->product_details['name_ar']) ? $this->product_details['name_ar'] : null,
                    'en' => isset($this->product_details['name_en']) ? $this->product_details['name_en'] : $this->product_details['name_ar'],
                ],
                'description' => [
                    'ar' => isset($this->product_details['description_ar']) ? $this->product_details['description_ar'] : null,
                    'en' => isset($this->product_details['description_en']) ? $this->product_details['description_en'] : null,
                ],
            ]),
        ]);
    }
}
