<?php

namespace App\Http\Resources;

use App\Models\Bill;
use Illuminate\Http\Resources\Json\JsonResource;

class BillResource extends JsonResource
{
    /**
     * Transform the resource into an array.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return array|\Illuminate\Contracts\Support\Arrayable|\JsonSerializable
     */
    public function toArray($request)
    {
        // return parent::toArray($request);
        return [
            'id' => $this->id,
            'code' => $this->code,
            'customer' => $this->whenLoaded('billable', function () {
                return $this->getCustomer();
            }),
            'release_date' => $this->release_date,
            "amount_paid" => number_format($this->amount_paid, 2),
            "remain" => number_format($this->remain, 2),
            'type' => $this->type,
            "before_discount" => number_format($this->before_discount, 2),
            "total_discount" => number_format($this->total_discount, 2),
            // "discount_on_bill" => number_format($this->discount_on_bill, 2),
            "tax_amount" => number_format($this->tax_amount, 2),
            'branch' => new BranchResource($this->whenLoaded('branch')),
            'warehouse' => new WarehouseResource($this->whenLoaded('warehouse')),
            // 'delivery' => new DeliveryResource($this->whenLoaded('delivery')),
            'attachment_file' => $this->attachment_file ? Bill::FILE_PATH . '/' . $this->attachment_file : null,
            'note' => $this->note,
            'status' => $this->status,
            'date_of_supply' => $this->date_of_supply,
            'payment_method' => $this->whenLoaded('billPaymentMethods', function () {
                return $this->getPaymentMethods();
            }),
            'sell_nature' =>  $this->whenLoaded('sellNature', function () {
                return $this->getSellNature();
            }),
            'bill_products' => BillProductResource::collection($this->whenLoaded('billProducts')),
            'bill_services' => BillServiceResource::collection($this->whenLoaded('billServices')),
            'journals' => JournalResource::collection($this->whenLoaded('journals')),
            'expenses' => BillExpenseResource::collection($this->whenLoaded('billExpenses')),
            'responsible' => $this->getWorker(),
            'time' => $this->created_at_time,
            'is_returned' => $this->is_returned,
            // 'returned_bill' => $this->returned_bill,
        ];
    }

    private function getCustomer()
    {
        if ($this->billable_type == 'App\\Models\\Supplier') {
            return new SupplierResource($this->billable_type::find($this->billable_id, ['id', 'name', 'phone']));
        }
        return new ClientResource($this->billable_type::find($this->billable_id, ['id', 'name', 'phone']));
    }

    private function  getPaymentMethods()
    {
        $resources = [];

        foreach ($this->billPaymentMethods as $paymentMethod) {
            if (!$paymentMethod->resource) {
                continue;
            }
            $resources[] =  [
                'paymentMethod' => new PaymentMethodResource($paymentMethod->paymentMethod),
                'model' => new $paymentMethod->resource(
                    $paymentMethod->payment_type::find($paymentMethod->payment_id)
                ),
                'amount' => $paymentMethod->amount
            ];
        }
        return  $resources;
    }

    private function getSellNature()
    {
        if ($this->sellNature) {
            $data = [
                'sell_nature_id' => $this->sellNature->main_sell_nature_id,
                'value' => $this->sellNature->value,
            ];
            $resource = $this->sellNature->resource;
            if ($resource) {
                $data['model'] = new $resource($this->sellNature->selling_type::find($this->sellNature->selling_id));
            }
        }
        return $data ?? null;
    }

    private function getWorker()
    {
        return $this->worker?->employee?->name;
    }
}
