<?php

namespace App\Http\Services\Account;

use App\Models\JournalReptation;
use App\Models\Shift;
use App\Models\Branch;
use App\Models\Account;
use App\Models\Company;
use App\Models\Journal;
use App\Models\Employee;
use App\Models\Warehouse;
use App\Http\Helper\CompanyHelper;
use App\Models\Client;
use App\Models\JournalRepeationNotification;
use App\Models\Supplier;
use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasOneThrough;
use Illuminate\Validation\ValidationException;
use App\Trait\UploadFileTrait;
use Carbon\Carbon;

class JournalService
{
    use UploadFileTrait;
    public static function createJournal(
        string $date,
        string $type,
        ?string $source = null,
        ?string $description,
        ?string $file = null,
        ?Employee $employee = null,
        string $status,
        array $debit = [],
        array $credit = [],
        ?Model $journalable = null,
        ?string $reference_code = null,
        ?Branch $branch = null,
        ?Warehouse $warehouse = null,
        ?bool $repeatable = null,
        ?array $repeatableData = [],
    ): ?Journal {
        // Begin a database transaction
        DB::beginTransaction();

        try {
            // dd($debit);
            if ($status != 'draft') {
                $credit_total = AccountService::updateMultiAccountAmountsWithParentAccounts($credit);
                $debit_total = AccountService::updateMultiAccountAmountsWithParentAccounts($debit);
            
                if (!self::isBalanced(debitTotal: $debit_total, creditTotal: $credit_total)) {
                    throw ValidationException::withMessages(['journal' => [__('messages.debit_credit_amounts_must_be_equal', ['type' => $type])]]);
                }
                if ($debit_total == 0 || $credit_total == 0) {
                    DB::commit();
                    return null;
                }
            }
            // Update accounts in the balance sheet


            // dd($credit_total,  $debit_total);
            // Check if debit equals credit, otherwise rollback

            // If both sides are zero, commit the transaction and return

            // Create the journal
            $journal_data = [
                'company_id' => CompanyHelper::getCompany(request())->id,
                'date' => $date,
                'type' => $type,
                'source' => $source,
                'description' => $description,
                'file' => $file,
                'employee_id' => optional($employee)->id,
                'branch_id' => optional($branch)->id,
                'warehouse_id' => optional($warehouse)->id,
                'status' => $status,
                'reference_code' => $reference_code,
                'shift_id' => self::getCurrentShift()?->id,
            ];

            // Use the provided relation if available, otherwise create a new journal
            $journal = $journalable ? $journalable->journals()->create($journal_data) : Journal::create($journal_data);
         

            // Create entries
            self::createEntries(debit: $debit, credit: $credit, journal: $journal);

            //Create Journal Repeation if exists

            if ($repeatable) {
                $journalRepeation =   JournalReptation::create(
                    [
                        'journal_id' => $journal->id,
                        'company_id' => CompanyHelper::getId(),
                        'counter' => 0,
                        'desc' => $description
                    ]
                        + $repeatableData
                );
                if ($journalRepeation->start_date < Carbon::now()) {
                    JournalRepeationNotification::create(
                        [
                            'company_id' => $journalRepeation->company_id,
                            'journal_repeation_id' => $journalRepeation->id,
                            'date' => $journalRepeation->start_date
                        ]
                    );
                }
            }

            // Commit the transaction
            DB::commit();

            return $journal;
        } catch (\Exception $e) {
            // Rollback in case of an exception
            DB::rollBack();
            throw $e;
        }
    }

  protected static function createEntries($debit, $credit, $journal)
    {
        // Create the journal debit entries
        // dd($credit);
        foreach ($debit as $debit_entry) {
            self::createEntry(journal: $journal, entry: $debit_entry, type: 'debit');
        }

        // Create the journal credit entries
        foreach ($credit as $credit_entry) {
            self::createEntry(journal: $journal, entry: $credit_entry, type: 'credit');
        }
    }

    protected static function createEntry(Journal $journal, $entry, $type)

    {
        // dd( Account::find($entry['account']->id));

        // $account = $entry['account'] instanceof \Illuminate\Database\Eloquent\Relations\HasOneThrough ? $entry['account']->first() : \App\Models\Account::find($entry['account']?->id);
        $account = $entry['account'] instanceof \Illuminate\Database\Eloquent\Relations\HasOneThrough 
    ? $entry['account']->first() 
    : ($entry['account'] ? \App\Models\Account::find($entry['account']->id) : null);
        // $account =   \App\Models\Account::find($entry['account']->id) ;

        if ($entry['amount'] > 0) {
            $journal->entries()->create([
                'status' => $type,
                'account_id' => $account->id?? null,
                'type_type' => $entry['type']['type'],
                'type_id' => $entry['type']['id'],
                'amount' => $entry['amount'],
                'note' =>  isset($entry['note']) ? $entry['note'] : null,
                'order' =>   isset($entry['order']) ? $entry['order'] : 0,
            ]);
        }
    }

    protected static function isBalanced($debitTotal, $creditTotal): bool
    {
        return (int)$debitTotal == (int)$creditTotal || (int)$debitTotal == -(int)$creditTotal;
    }

    public static function handleSide($data, $side_type)
    {
        $side_data = [];
        $side_amount = 0;
        foreach ($data as $key => $row) {
            $side_data[$key]['account'] = Account::findOrFail($row['account']);
            $side_data[$key]['type']['type'] = self::getModelFormEntryRef($row['ref']['type'] ?? null);
            $side_data[$key]['type']['id'] = $row['ref']['id'] ?? null;
            $side_data[$key]['amount'] = $row['amount'];
            $side_data[$key]['entry_type'] = $side_type;
            $side_data[$key]['note'] = $row['note'];
            $side_data[$key]['order'] = isset($row['order']) ? $row['order'] : 0;
            $side_amount += $row['amount'];
        }
        return ["side_data" => $side_data, "side_amount" => $side_amount];
    }


    private static function getModelFormEntryRef($ref)
    {
        switch ($ref) {
            case 'client':
                return Client::class;
                break;
            case 'supplier':
                return Supplier::class;
                break;

            default:
                null;
                break;
        }
    }
    private static function getCurrentShift(): ?Shift
    {
        return request()->user() instanceof Company
            ? null
            : request()->user()?->shifts()?->whereNull('close_name')?->latest()?->first();
    }



    protected static function getEntriesFromJournal($journal)
    {

        $entries = [];
        foreach ($journal->entries as $journalEntry) {

            if ($journalEntry->status == 'credit') {
                $entries['credit'][] = [
                    'account' => Account::find($journalEntry->account_id),
                    'type' => [
                        'type' => $journalEntry->type_type,
                        'id' => $journalEntry->type_id,
                    ],
                    'amount' => $journalEntry->amount,
                    'entry_type' => "credit",
                ];
            } else {
                $entries['debit'][] = [
                    'account' => Account::find($journalEntry->account_id),
                    'type' => [
                        'type' => $journalEntry->type_type,
                        'id' => $journalEntry->type_id,
                    ],
                    'amount' => $journalEntry->amount,
                    'entry_type' => "debit",
                ];
            }
        }

        return $entries;
    }

    public static function deleteJournal($journal)
    {

        try {
            DB::beginTransaction();
            if (!$journal->entries->isEmpty()) {
                if ($journal->status != 'draft') {
                    self::applyReversedEntries(self::getEntriesFromJournal($journal));
                }
                $journal->entries()->delete();
            }
            $journal->journalRepeation()->each(function ($repeation) {
                $repeation->notifications()->delete();
                $repeation->delete();
            });


            $journal->delete();

            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();

            throw $e;
        }
    }

    public  function updateJournal(
        $request,
        $journal,
         $journalRepeation = null,
        ?bool $repeatable = null,
        ?array $repeatableData = [],
        String $status
    ) {

        $debit_side = self::handleSide(
            self::filterAccountsSide(
                accounts: $request->accounts,
                side: 'debit'
            ),
            'debit'
        );

        $credit_side = self::handleSide(
            self::filterAccountsSide(
                accounts: $request->accounts,
                side: 'credit'
            ),
            'credit'
        );
        if ($debit_side['side_amount'] !== $credit_side['side_amount']) {
            throw ValidationException::withMessages(['debit' => [__('messages.journal_validation_error')]]);
        }
        $file = $this->uploadFile(Journal::UPLOADED_FILES, $request->file, $journal->file);
        $validatedData = $request->validated();
        unset($validatedData['accounts']);
        unset($validatedData['file']);
        try {
            DB::beginTransaction();
            //update journal data itself

            //remove old entries

            if ($status != 'draft') {
                if ($journal->status != 'draft') {
                    if (!$journal->entries->isEmpty()) {
                        self::applyReversedEntries(self::getEntriesFromJournal($journal));
                    }
                }

                $debit_total = AccountService::updateMultiAccountAmountsWithParentAccounts($debit_side['side_data']);
                $credit_total = AccountService::updateMultiAccountAmountsWithParentAccounts($credit_side['side_data']);
                // Check if debit equals credit, otherwise rollback
                if (!self::isBalanced(debitTotal: $debit_total, creditTotal: $credit_total)) {
                    throw ValidationException::withMessages(['journal' => [__('messages.debit_credit_amounts_must_be_equal', ['type' => $journal->type])]]);
                }

                // If both sides are zero, commit the transaction and return
                if ($debit_total == 0 || $credit_total == 0) {
                    DB::commit();
                    return null;
                }
            }

            $journal->entries()->delete();


            self::createEntries(debit: $debit_side['side_data'], credit: $credit_side['side_data'], journal: $journal);

            if ($repeatable) {
                $journal->journalRepeation()->updateOrCreate(
                    [
                        'journal_id' => $journal->id,
                        'company_id' => CompanyHelper::getId(),
                        'counter' => 0,

                    ]
                        + $repeatableData
                );
            }
            
             if($request->type=="repeat"){
                $journalRepeation->update([
                    'journal_id' => $journal->id,
                    'company_id' => CompanyHelper::getId(),
                    'counter' => 0,
                    'desc' => $request->desc,
                    'start_date' => $request->start_date,
                    'end_date' => $request->end_date,
                    'period' => $request->period,
                ]);

            }
            $journal->update([
                'file' => $file,
            ] +  $validatedData);

            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();

            throw $e;
        }
    }
    public static function reverseJournal(Journal $journal)
    {
        $entries = self::getEntriesFromJournal($journal);
        $journal->update(['status' => 'accredited reversed']);
        self::createJournal(
            date: now(),
            type: $journal->type,
            source: $journal->source,
            file: $journal->file,
            description: $journal->description,
            employee: $journal->employee,
            status: 'reversed',
            journalable: $journal->journalable,
            reference_code: $journal->reference_code,
            branch: $journal->branch,
            warehouse: $journal->warehouse,
            debit: $entries['credit'],
            credit: $entries['debit'],
            repeatable: null,
            repeatableData: null,
        );
    }
    protected static function applyReversedEntries($entries)
    {

        foreach ($entries['debit'] as $debit_entry) {
            $operator = AccountService::getOperator($debit_entry['account'], 'credit');
            AccountService::updateAccountAmountWithParentAccounts(
                account: $debit_entry['account'],
                amount: $debit_entry['amount'],
                operator: $operator
            );
        }


        foreach ($entries['credit'] as $credit_entry) {
            $operator = AccountService::getOperator($credit_entry['account'], 'debit');
            AccountService::updateAccountAmountWithParentAccounts(
                account: $credit_entry['account'],
                amount: $credit_entry['amount'],
                operator: $operator
            );
        }
    }

    public static function filterAccountsSide(array $accounts, string $side): array
    {
        return array_filter($accounts, fn($account) => $account['type'] === $side);
    }
}
