<?php

namespace App\Http\Services\Account;

use App\Http\Helper\CompanyHelper;
use App\Models\Branch;
use App\Models\Product;
use App\Models\Warehouse;
use App\Models\ProductUnit;
use App\Models\ProductWarehouse;
use App\Jobs\ProductTrackQuantity;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use App\Http\Services\Account\OpenBalanceJournalService;
use App\Models\OpenBalance;
use App\Models\OpenBalanceProduct;
use App\Models\ProductQuantityTrack;

class OpenBalanceService
{

    private static function updateWarehouseQuantity(
        ProductWarehouse $productWarehouse,
        ProductUnit $productUnit,
        float $newQuantity,
        ?float $oldQuantity = null,
        float $price
    ): float {
        // Calculate quantity in main unit
        $convertedNewQuantity = $productUnit->conversion_factor * $newQuantity;

        if ($oldQuantity === null) {
            // Case: Product does not exist in open balance (initial creation)
            $quantity = $convertedNewQuantity;
            $productWarehouse->update([
                'value' => $productWarehouse->value ? $productWarehouse->value + ($quantity * $price) : $quantity * $price,
                'quantity' => $productWarehouse->quantity ? $productWarehouse->quantity + $quantity : $quantity,
                'price' => $price,
                'open_balance_quantity' => $quantity,
                'open_balance_price' => $price,
            ]);
            return $quantity;
        }

        // Calculate the difference between the new and old quantities
        $quantityDifference = $convertedNewQuantity - $oldQuantity;

        if ($quantityDifference == 0) {
            // Case: No change in quantity, update price-related fields

            $oldPricePerOne = $productWarehouse->open_balance_price / $productWarehouse->open_balance_quantity;
            $oldValue = $oldPricePerOne * $oldQuantity;
            $newValue = $productWarehouse->value ? $productWarehouse->value + ($newQuantity * $price) - $oldValue : ($newQuantity * $price) - $oldValue;

            $productWarehouse->update([
                'value' => $newValue,
                'price' => $price,
                'open_balance_price' => $price,
            ]);
            return $convertedNewQuantity;
        }

        if ($quantityDifference > 0) {
            // Case: Quantity increased
            $newValue = self::calculateNewValue($productWarehouse, $convertedNewQuantity, $oldQuantity, $price);
            $productWarehouse->update([
                'value' => $newValue,
                'quantity' => $productWarehouse->quantity ? $productWarehouse->quantity + $quantityDifference : $quantityDifference,
                'price' => $price,
                'open_balance_quantity' => $convertedNewQuantity,
                'open_balance_price' => $price,
            ]);
            return $convertedNewQuantity;
        }

        // Case: Quantity decreased
        if ($productWarehouse->quantity + $quantityDifference < 0) {
            $minimumQuantity = abs($quantityDifference + $productWarehouse->quantity) + $convertedNewQuantity;
            $productName = $productWarehouse->product->name;
            throw ValidationException::withMessages([
                "message" => "  $productName  لا يمكن أن نكون كمية  اقل من $minimumQuantity في المخزن "
            ]);
        }

        $newValue = self::calculateNewValue($productWarehouse, $convertedNewQuantity, $oldQuantity, $price);
        $productWarehouse->update([
            'value' => $newValue,
            'quantity' => $productWarehouse->quantity ? $productWarehouse->quantity + $quantityDifference : $quantityDifference,
            'price' => $price,
            'open_balance_quantity' => $convertedNewQuantity,
            'open_balance_price' => $price,
        ]);

        return $convertedNewQuantity;
    }

    private static function calculateNewValue(
        ProductWarehouse $productWarehouse,
        float $convertedNewQuantity,
        float $oldQuantity,
        float $price
    ): float {

        $oldValue = $productWarehouse->open_balance_price * $oldQuantity;
        // dd($productWarehouse->value , $oldValue);
        $productWarehouseValue = $productWarehouse->value;
        // If the current value is equal to the old value, reset it to 0
        if ($productWarehouseValue === $oldValue) {
            $productWarehouseValue = 0;
        } else {
            // Subtract the old value from the current product value
            $productWarehouseValue -= $oldValue;
        }
        // dd($productWarehouseValue + ($convertedNewQuantity * $price));
        // Add the new value to the product value
        return $productWarehouseValue + ($convertedNewQuantity * $price);
    }
    private static function getProductWithWarehouseAndUnit(int $productId, int $unitId, int $warehouseId): Product
    {
        return Product::where('id', $productId)
            ->with(
                [
                    'warehouses.warehouse:id',
                    'warehouses' => fn($query) => $query->where('warehouse_id', $warehouseId),
                    'units.unit:id',
                    'units' => fn($query) => $query->where('unit_id', $unitId),
                ]
            )
            ->first();
    }

    public static function createOpenBalance(array $products, $branchId, $warehouseId, $date, $description, $status): void
    {

        DB::beginTransaction();

        $openBalance = OpenBalance::create([
            'branch_id' => $branchId,
            'company_id' => CompanyHelper::getId(),
            'warehouse_id' => $warehouseId,
            'description' => $description,
            'date' => $date,
            'status' => $status,
        ]);
        foreach ($products as $newProduct) {
            $product = self::getProductWithWarehouseAndUnit(
                productId: $newProduct['product_id'],
                unitId: $newProduct['unit_id'],
                warehouseId: $warehouseId
            );

            $productUnit = $product->units()->where('unit_id', $newProduct['unit_id'])->first();
            $productWarehouse = $product->warehouses()->where('warehouse_id', $warehouseId)->first();
            $productWarehouse = self::openBalanceCheckers(
                product: $product,
                productUnit: $productUnit,
                warehouseId: $warehouseId,
                productId: $newProduct['product_id'],
                productWarehouse: $productWarehouse
            );


            $quantity = $newProduct['quantity'];
            $price = $newProduct['price'];
            if ($openBalance->status != 'draft') {
                $quantityWithMainUnit = self::updateWarehouseQuantity(
                    productWarehouse: $productWarehouse,
                    productUnit: $productUnit,
                    newQuantity: $quantity,
                    oldQuantity: null,
                    price: $price
                );

                $journal = OpenBalanceJournalService::createJournal(
                    total: self::calcTotal(quantity: $quantity, price: $price),
                    product: $product,
                    productWarehouse: $productWarehouse,
                    account: Warehouse::find($warehouseId)->account,
                    date: $date,
                    branch: Branch::findOr($branchId, fn() => null),
                    warehouse: Warehouse::findOr($warehouseId, fn() => null),
                    status: $openBalance->status
                );

                ProductTrackQuantity::dispatch(
                    product: $product,
                    journal: $journal,
                    productUnit: $productUnit,
                    quantity: $quantityWithMainUnit,
                    productWarehouse: $productWarehouse,
                );
            }
            OpenBalanceProduct::create([
                'open_balances_id' => $openBalance->id,
                'journal_id' => $journal->id ?? null,
                'product_id' => $newProduct['product_id'],
                'unit_id' => $productUnit->unit_id,
                'quantity' => $quantity,
                'cost' => $price,
            ]);
        }

        DB::commit();
    }

    private static function alreadyHasOpenBalance(Product $product, ProductWarehouse $productWarehouse): bool
    {
        return $product->openBalanceJournal($productWarehouse)->exists();
    }

    private static function createProductWarehouse(int $productId, int $warehouseId): ProductWarehouse
    {
        return ProductWarehouse::create([
            'warehouse_id' => $warehouseId,
            'product_id' => $productId,
        ]);
    }

    private static function openBalanceCheckers(
        Product $product,
        ProductUnit $productUnit,
        int $warehouseId,
        int $productId,
        ?ProductWarehouse $productWarehouse = null,
    ): ProductWarehouse {
        if (!$productWarehouse) {
            $productWarehouse = self::createProductWarehouse(
                productId: $productId,
                warehouseId: $warehouseId
            );
        }

        if (!$productUnit) {
            DB::rollBack();
            throw ValidationException::withMessages([
                'unit_id' => [__('messages.product_does_not_have_this_unit')]
            ]);
        }

        if (self::alreadyHasOpenBalance($product, $productWarehouse)) {
            DB::rollBack();
            throw ValidationException::withMessages([
                'product_id' => [__('messages.product_already_has_open_balance_in_this_warehouse')]
            ]);
        }
        return $productWarehouse;
    }

    private static function calcTotal(int $quantity, float $price): float
    {
        return $quantity * $price;
    }
    public static function updateOpenBalance($data, OpenBalance $openBalance)
    {
        DB::beginTransaction();
        if ($openBalance->status == 'accredited') {
            throw ValidationException::withMessages([
                'openBalance' => 'can not update open balance'
            ]);
        }


        foreach ($data['products'] as $newProduct) {
            $product = self::getProductWithWarehouseAndUnit(
                productId: $newProduct['product_id'],
                unitId: $newProduct['unit_id'],
                warehouseId: $data['warehouse_id'],
            );

            $productUnit = $product->units()->where('unit_id', $newProduct['unit_id'])->first();
            $productWarehouse = $product->warehouses()->where('warehouse_id', $data['warehouse_id'])->first();
            $productWarehouse = self::openBalanceCheckers(
                product: $product,
                productUnit: $productUnit,
                warehouseId: $data['warehouse_id'],
                productId: $newProduct['product_id'],
                productWarehouse: $productWarehouse
            );

            $price = $newProduct['price'];
            $quantity = $newProduct['quantity'];
            $oldProduct = self::getOldProduct($openBalance, $newProduct);
            if ($data['status'] != 'draft') {

                $quantityWithMainUnit = self::updateWarehouseQuantity(
                    productWarehouse: $productWarehouse,
                    productUnit: $productUnit,
                    newQuantity: $newProduct['quantity'],
                    oldQuantity: $data['status'] != 'draft' && $openBalance->status == 'draft' ? null : $oldProduct?->quantity/*  */,
                    price: $price
                );
                self::deleteOldProductAndItsDependencies($oldProduct);

                $journal = OpenBalanceJournalService::createJournal(
                    total: self::calcTotal(quantity: $quantity, price: $price),
                    product: $product,
                    productWarehouse: $productWarehouse,
                    account: Warehouse::find($data['warehouse_id'])->account,
                    date: $data['date'],
                    branch: Branch::findOr($data['branch_id'], fn() => null),
                    warehouse: Warehouse::findOr($data['warehouse_id'], fn() => null),
                    status: $data['status']
                );
                ProductTrackQuantity::dispatch(
                    product: $product,
                    journal: $journal,
                    productUnit: $productUnit,
                    quantity: $quantityWithMainUnit,
                    productWarehouse: $productWarehouse,
                );
            }
            $openBalance->update(
                [
                    'date' => $data['date'],
                    'description' => $data['description'],
                    'status' => $data['status'],
                ]
            );
            $openBalance->products()
                ->where('product_id', $newProduct['product_id'])
                ->delete();
            OpenBalanceProduct::create([
                'open_balances_id' => $openBalance->id,
                'journal_id' => $journal->id ?? null,
                'product_id' => $newProduct['product_id'],
                'unit_id' => $productUnit->unit_id,
                'quantity' => $quantity,
                'cost' => $price,
            ]);
        }
        DB::commit();
    }



    public static function deleteOpenBalance(OpenBalance $openBalance)
    {
        if ($openBalance->status == 'accredited') {
            throw ValidationException::withMessages([
                'openBalance' => 'can not delete accredited open balance'
            ]);
        }
        DB::beginTransaction();
        self::deleteOpenBalanceProducts($openBalance);
        $openBalance->delete();
        DB::commit();
    }

    private static function deleteOpenBalanceProducts(OpenBalance $openBalance)
    {
        foreach ($openBalance->products as $openBalanceProduct) {
            ProductQuantityTrack::where('journal_id', $openBalanceProduct->journal_id)->delete();
            OpenBalanceProduct::find($openBalanceProduct->id)->delete();
            if ($openBalance->status != 'draft') {
                JournalService::deleteJournal($openBalanceProduct->journal);
            }
        }
    }

    private static function getOldProductQuantity(OpenBalance $openBalance, $newProduct)
    {
        return $openBalance->products()->where([

            ['product_id', $newProduct['product_id']],

            ['unit_id', $newProduct['unit_id'],]

        ])->first('quantity');
    }
    private static function getOldProduct(OpenBalance $openBalance, $newProduct)
    {
        return $openBalance->products()->where([

            ['product_id', $newProduct['product_id']],

            ['unit_id', $newProduct['unit_id'],]

        ])->first();
    }
    private static function deleteOldProductAndItsDependencies($oldProduct)
    {
        if ($oldProduct) {
            ProductQuantityTrack::where('journal_id', $oldProduct->journal_id)->delete();
            OpenBalanceProduct::find($oldProduct->id)->delete();
            if ($oldProduct->journal) {
                JournalService::deleteJournal($oldProduct->journal);
            }
        }
    }
}
