<?php

namespace App\Http\Services\Bound;

use App\Models\Bound;
use App\Models\Branch;
use App\Models\Journal;
use App\Models\Bill;
use App\Models\Warehouse;
use App\Trait\BoundHelperTrait;
use Illuminate\Support\Facades\DB;
use App\Http\Requests\Api\Bound\BoundRequest;
use App\Http\Services\Account\JournalService;
use App\Models\BoundEntries;
use Illuminate\Validation\ValidationException;


class BoundService
{
    use BoundHelperTrait;

    private Bound $bound;

    public function create(BoundRequest $request, $type)
    {
        DB::beginTransaction();

        $validatedData = $request->all();
        $entries =   $validatedData['entries'];
        unset($validatedData['entries']);


        // create bound.

        $this->bound = Bound::create($validatedData);
        $this->discount =$validatedData['discount'] ?? 0;
        //create bound entries
        foreach ($entries as $entry) {
            BoundEntries::create([
                'bound_id' => $this->bound->id
            ] + $entry);
        }
        // create journal for bound.

        $this->createJournal(
            branch: Branch::findOr($request->branch_id, fn() => null),
            warehouse: Warehouse::findOr($request->warehouse_id, fn() => null),
            boundEntries: $this->bound->boundEntries->all(),
            type: $type,
            status: $validatedData['status'],
        );
        
        if($this->bound->status=='accredited'){
            $this->updateBill($request);
        }        

        DB::commit();

        return $this->bound;
    }

    public function update(BoundRequest $request, Bound  $bound, $type)
    {
        $this->preventActionOnAccreditedBound($bound);
        DB::beginTransaction();

        $validatedData = $request->all();
        $entries =   $validatedData['entries'];

        unset($validatedData['entries']);
        $this->bound = $bound;
        //delete old entries and journal
        $this->deleteBoundDependencies($this->bound);
        //new entries and update bound status


        $this->bound->update($validatedData);
        foreach ($entries as $entry) {
            BoundEntries::create([
                'bound_id' => $this->bound->id
            ] + $entry);
        }
        //new journal
        $this->bound->load('boundEntries');

        $this->createJournal(
            branch: Branch::findOr($request->branch_id, fn() => null),
            warehouse: Warehouse::findOr($request->warehouse_id, fn() => null),
            boundEntries: $this->bound->boundEntries->all(),
            type: $type,
            status: $validatedData['status'],
        );
        if($this->bound->status=='accredited'){
            $this->updateBill($request);
        }          

        DB::commit();
    }

    public function delete($bound)
    {
        $this->preventActionOnAccreditedBound($bound);
        $this->bound = $bound;
        $this->deleteBoundDependencies($this->bound);
        $this->bound->delete();
    }

    private function deleteBoundDependencies(Bound $bound)
    {

        $this->bound->boundEntries()->delete();
        JournalService::deleteJournal($bound->journals()->first());
    }

    private  function preventActionOnAccreditedBound($bound)
    {
        if ($bound->status == 'accredited') {
            throw ValidationException::withMessages([
                'bound' => [__('messages.accredited_bound_cannot_delete_or_update_')]
            ]);
        }
    }
    
    // public function updateBill($request)
    // {
    //     if ($request->bill_id) {
    //         $bill = Bill::findOrFail($request->bill_id);
    //         $totalValue = collect($request->entries)->sum('value');
    
    //         $bill->update([
    //             "amount_paid" => $bill->amount_paid + $totalValue,
    //             "remain" => $bill->remain - $totalValue,
    //         ]);
    //     }
       
    // }  
    public function updateBill($request)
    {
        if ($request->has('bills') && is_array($request->bills)) {
            foreach ($request->bills as $billData) {
                if (isset($billData['bill_id']) && isset($billData['value'])) {
                    $bill = Bill::findOrFail($billData['bill_id']);
                    $billDiscount= $billData['discount'] ?? 0;
                    
                    $this->bound->bills()->attach($billData['bill_id'], [
                        'amount'     => $billData['value'],
                        'discount'   => $billDiscount,
                        // $this->discount ?? 0,
                    ]);
                    
                    $bill->update([
                        'amount_paid' => $bill->amount_paid + $billData['value'] + $billDiscount,
                        'remain'      => $bill->remain - $billData['value'] - $billDiscount,
                    ]);
                }
            }
        }
    }
    
    public function toggleArchive(Bound $bound): void
    {
        $bound->trashed() ? $bound->restore() : $bound->delete();
    }
}
