<?php

namespace App\Http\Services;

use App\Http\Helper\AccountHelper;
use App\Http\Services\Account\AccountService;
use App\Http\Services\Account\JournalService;
use App\Jobs\ProductTrackQuantity;
use App\Models\Journal;
use App\Models\Product;
use App\Models\ProductUnit;
use App\Models\Warehouse;
use Illuminate\Support\Facades\DB;

class ProductInventoryService
{
    public static function update(array $products, Warehouse $warehouse)
    {

        DB::beginTransaction();

        $lostAmount = 0;
        $earnedAmount = 0;

        foreach ($products as $product) {

            $productWarehouse = $warehouse->products()->where('product_id', $product['id'])->first();

            $quantity = $product['actual_quantity'] - $productWarehouse->quantity;

            if ($quantity <= 0) {

                $lostAmount += self::getProductPrice($productWarehouse, $product['id']) *  abs($quantity);
            } else {

                $earnedAmount += self::getProductPrice($productWarehouse, $product['id']) * abs($quantity);
            }
            $products[] = $productWarehouse;
            $productWarehouse->update([
                'quantity' =>  $product['actual_quantity']
            ]);
        }
        if ($lostAmount > 0) {
            $LostEntries = self::createLostEntries($warehouse, $lostAmount);

            $journal = JournalService::createJournal(
                date: now(),
                type: __('constants.auto_type'),
                source: 'Products Inventory',
                description: __('messages.product_settlements'),
                file: null,
                employee: null,
                status: true,
                debit: $LostEntries['debit'],
                credit: $LostEntries['credit'],
                journalable: null,
                branch: null,
                warehouse: $warehouse,
            );
            // self::trackProductQuantity($products, $journal, $quantity, $warehouse);
        }
        if ($earnedAmount > 0) {
            $earnedEntries = self::createEarnedEntries($warehouse, $earnedAmount);

            $journal = JournalService::createJournal(
                date: now(),
                type: __('constants.auto_type'),
                source: 'Products Inventory',
                description: __('messages.product_settlements'),
                file: null,
                employee: null,
                status: true,
                debit: $earnedEntries['debit'],
                credit: $earnedEntries['credit'],
                journalable: null,
                branch: null,
                warehouse: $warehouse,
            );
            // self::trackProductQuantity($products, $journal, $quantity, $warehouse);
        }


        DB::commit();
    }
    private static function getProductPrice($productWarehouse): float
    {
        return  ProductUnit::where('product_id', $productWarehouse->product_id)->where('main', true)->orWhere('conversion_factor', 1.0)->first()->purchasing_price ?? 0.0;
    }
    private static function createLostEntries($warehouse, $lostAmount): array
    {
        $entries = [
            'debit' => [],
            'credit' => [],
        ];

        $entries['debit'][] =
            [
                'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_INVENTORY_LOSSES, 'Inventory'),
                'type' => [
                    'type' => null,
                    'id' => null,
                ],
                'amount' => $lostAmount,
                'entry_type' => "debit",
            ];

        $entries['credit'][] =
            [
                'account' => $warehouse->account,
                'type' => [
                    'type' => null,
                    'id' => null,
                ],
                'amount' => $lostAmount,
                'entry_type' => "debit",
            ];
        return $entries;
    }
    private static function createEarnedEntries($warehouse, $earnedAmount): array
    {
        $entries = [
            'debit' => [],
            'credit' => [],
        ];

        $entries['debit'][] =
            [
                'account' => $warehouse->account,
                'type' => [
                    'type' => null,
                    'id' => null,
                ],
                'amount' => $earnedAmount,
                'entry_type' => "debit",
            ];

        $entries['credit'][] =
            [
                'account' => AccountService::getAccountForModel(AccountHelper::ACCOUNT_INVENTORY_REVENUES, 'Inventory'),
                'type' => [
                    'type' => null,
                    'id' => null,
                ],
                'amount' => $earnedAmount,
                'entry_type' => "debit",
            ];
        return $entries;
    }
    private  static function trackProductQuantity(array $products, Journal $journal, $quantity, $warehouse)
    {
        foreach ($products as $productData) {
            $product = Product::find($productData['id']);


            // Fetch the main or default product unit
            $productUnit = ProductUnit::where('product_id', $productData['id'])
                ->where(function ($query) {
                    $query->where('main', true)
                        ->orWhere('conversion_factor', 1.0);
                })
                ->first();

            // Fetch the product warehouse data
            $productWarehouse = $warehouse->products()->where('product_id', $productData['id'])->first();
            dd(   $productWarehouse);

            // Dispatch the job
            ProductTrackQuantity::dispatch(
                product: $product,
                journal: $journal,
                productUnit: $productUnit,
                quantity: -$productWarehouse['quantity'], // Use the quantity from $productData
                productWarehouse: $productWarehouse
            );
        }
    }
}
