<?php

namespace App\Models;

use App\Http\Resources\AccountResource;
use Illuminate\Database\Eloquent\Model;
use Spatie\Translatable\HasTranslations;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Staudenmeir\LaravelAdjacencyList\Eloquent\HasRecursiveRelationships;


class Account extends Model
{
    use HasFactory, HasTranslations, SoftDeletes, HasRecursiveRelationships;

    const ROLE = [
        'name' => 'nullable|array',
        'description' => 'nullable|array',
        'name_ar' => 'required|string',
        'name_en' => 'nullable|string',
        'description_ar' => 'nullable|string',
        'description_en' => 'nullable|string',
        'active' => 'required|boolean',
    ];
    protected $fillable = [
        'name',
        'description',
        'code',
        'parent_id',
        'type', // 'debit -credit'
        'status', // 'main -sub'
        'company_id',
        'active',
        'amount',
        'level_id',
    ];

    public $translatable = [
        'name',
        'description',
    ];

    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    public function parent()
    {
        return $this->belongsTo(Account::class, 'parent_id')->with('parent');
    }

    public function level()
    {
        return $this->belongsTo(Level::class);
    }

    public function entries()
    {
        return $this->hasMany(JournalEntry::class)
            ->whereHas('journal', function ($query) {
                $query->where('status', '!=', 'draft');
            });
    }

    public function children()
    {
        return $this->hasMany(Account::class, 'parent_id')->with([
            'children.entries' => function ($query) {
                $fromDate = request()->query('from');
                $toDate = request()->query('to');

                if ($fromDate) {
                    $query->where('created_at', '>=', $fromDate);
                }

                if ($toDate) {
                    $query->where('created_at', '<=', $toDate);
                }
            },
        ])->when(request()->has('level'), fn($query) => $query->where('level_id', '<=', request()->input('level')));
    }

    public function models()
    {

        return $this->belongsToMany(MainModel::class, 'account_model_group', 'account_id', 'model_id');
    }

    public function setAmountAttribute($value)
    {
        $this->attributes['amount'] = $value * 100;
    }

    public function getAmountAttribute($value)
    {
        return $value / 100;
    }
    public function scopeArchived($query)
    {
        $query->onlyTrashed();
    }

    public function scopeCodeSearch($query, $code)
    {
        $query->where('code', 'LIKE', "%" . "$code" . "%");
    }

    public function scopeNameSearch($query, $name)
    {
        $query->where(
            'name->' . app()->getLocale(),
            'LIKE',
            "%" . "$name" . "%"
        );
    }
    public function custody()
    {
        return $this->hasMany(Custody::class);
    }
    public function custodyTaken()
    {
        return $this->hasMany(CustodyTaken::class);
    }
    public function resolveResource()
    {
        return AccountResource::class;
    }
    public function accountModelGroups()
    {
        return $this->hasMany(AccountModelGroup::class);
    }
}
