<?php

namespace App\Models;

use App\Http\Enums\Bill\BillEnum;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class ProductUnit extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'product_unit';

    protected $fillable = [
        'unit_id',
        'product_id',
        'main',
        'virtual_sale',
        'virtual_buying',
        'conversion_factor',
        'purchasing_price',
        'selling_price',
        'barcode',
        'tax_id',
        'weight',
    ];

    protected $with = ['unit'];

    const RULES = [
        'product_unit' => 'nullable|array',
        'product_unit.*.unit_id' => 'required|exists:company_units,id',
        'product_unit.*.weight' => 'nullable|in:on,off',
        'product_unit.*.main' => 'required|in:on,off',
        'product_unit.*.virtual_sale' => 'required|in:on,off',
        'product_unit.*.virtual_buying' => 'required|in:on,off',
        'product_unit.*.conversion_factor' => 'required|integer|min:1',
        'product_unit.*.purchasing_price' => 'required|numeric',
        'product_unit.*.selling_price' => 'required|numeric',
    ];

    protected $casts = [
        'weight' => 'boolean',
        'main' => 'boolean',
        'virtual_sale' => 'boolean',
        'virtual_buying' => 'boolean',
    ];

    public function unit()
    {
        return $this->belongsTo(CompanyUnit::class, 'unit_id');
    }

    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    public function warehouseUnit()
    {
        return $this->hasMany(WarehouseUnit::class);
    }

    public function lastSalesPrice()
    {
        return $this->hasOne(BillProduct::class, 'unit_id')
            ->join('bills', 'bill_products.bill_id', '=', 'bills.id')
            ->where('bills.type', BillEnum::sales)
            ->latest('bill_products.created_at');
    }

    public function lastPurchasePrice()
    {
        return $this->hasOne(BillProduct::class, 'unit_id')
            ->join('bills', 'bill_products.bill_id', '=', 'bills.id')
            ->where('bills.type', BillEnum::purchases)
            ->latest('bill_products.created_at');
    }
}
