<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Http\Resources\ProductWarehouseResource;
use Illuminate\Database\Eloquent\Factories\HasFactory;


class ProductWarehouse extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'warehouse_id',
        'product_id',
        'min_quantity',
        'max_quantity',
        'value',
        'price',
        'quantity',
        'actual_quantity',
        'open_balance_price',
        'open_balance_quantity',
    ];

    const RULES = [
        'warehouse_unit' => 'nullable|array',
        'warehouse_unit.*.warehouse_id' => 'required|exists:warehouses,id|distinct',
        'warehouse_unit.*.min_quantity' => 'nullable|numeric',
        'warehouse_unit.*.max_quantity' => 'nullable|numeric',
        'warehouse_unit.*.units' => 'required|array|min:1',
        'warehouse_unit.*.units.*.unit_id' => 'required|exists:company_units,id',
        'warehouse_unit.*.units.*.purchasing_price' => 'required|numeric|min:0',
        'warehouse_unit.*.units.*.selling_price' => 'required|numeric|min:0',
    ];

    public function resolveResource()
    {
        return ProductWarehouseResource::class;
    }

    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class, 'warehouse_id');
    }

    public function productQuantityTracks()
    {
        return $this->hasMany(ProductQuantityTrack::class);
    }

    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    public function unit()
    {
        return $this->belongsTo(CompanyUnit::class, 'unit_id');
    }

    public function warehouseUnits()
    {
        return $this->hasMany(WarehouseUnit::class);
    }

    public function journals()
    {
        return $this->morphMany(Journal::class, 'journalable');
    }

    public function getPriceAttribute($value)
    {
        return $value / 100;
    }

    public function setPriceAttribute($value)
    {
        $this->attributes['price'] = $value * 100;
    }

    public function getOpenBalancePriceAttribute($value)
    {
        return $value / 100;
    }

    public function setOpenBalancePriceAttribute($value)
    {
        $this->attributes['open_balance_price'] = $value * 100;
    }

    public function getValueAttribute($value)
    {
        return $value / 100;
    }

    public function setValueAttribute($value)
    {
        $this->attributes['value'] = $value * 100;
    }
}
