<?php

namespace App\Models;

use App\Models\Address;
use App\Rules\UniqueCompanyPhone;
use App\Http\Helper\AccountHelper;
use Illuminate\Database\Eloquent\Model;
use App\Http\Enums\User\UserStatusEnum;
use App\Http\Resources\SupplierResource;
use Spatie\Translatable\HasTranslations;
use Illuminate\Database\Eloquent\Builder;
use App\Http\Services\Account\AccountService;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Supplier extends Model
{
    use HasFactory, HasTranslations, SoftDeletes;

    const RULES = [
        'code' => 'required',
        'name' => 'nullable',
        'name_ar' => 'required|string',
        'name_en' => 'nullable|string',
        'phone_2' => 'nullable',
        // 'address_id' => 'required',
        'note' => 'nullable',
        'status' => 'required',
        'company_id' => 'required',
        'group_id' => 'nullable|exists:attributes,id,model,suppliers',
        'CRN' => 'nullable|string|max:50',
        'TRN' => 'nullable|string|max:50',
        'admins' => 'nullable|array',
        'admins.*.name' => 'required_with:admins|string|max:255',
        'admins.*.functional_role' => 'nullable|string|max:255',
        'admins.*.phone' => 'nullable|string|max:20',
        'admins.*.email' => 'nullable|email|max:255',
        'admins.*.note' => 'nullable|string',
        
        'payment_data' => 'nullable|array',
        'payment_data.bank_name' => ['nullable', 'string'],
        'payment_data.bank_account_owner_name' => ['nullable', 'string'],
        'payment_data.account_number' => ['nullable', 'string'],
        'payment_data.currency_id' => ['nullable', 'exists:company_currencies,id'],
        'payment_data.iban_code' => ['nullable', 'string'],
        'payment_data.swift_code' => ['nullable', 'string'],
        'payment_data.country_id' => ['nullable', 'exists:locations,id'],
        'payment_data.bank_address' => ['nullable', 'string'],

    ];

    protected $fillable = [
        'code',
        'name',
        'phone',
        'phone_2',
        'address_id',
        'note',
        'status',
        'company_id',
        'group_id',
        'CRN', // Commercial Registration No
        'TRN', // Tax registration number
        'whatsapp_number', 
        'website',
    ];

    public $translatable = ['name'];

    protected $casts = [
        'status' => UserStatusEnum::class,
    ];

    public function blocked()
    {
        return $this->hasOne(BlockSupplier::class);
    }

    public function address()
    {
        return $this->belongsTo(Address::class, 'address_id');
    }

    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    public function bills()
    {
        return $this->morphMany(Bill::class, 'billable');
    }

    public function entries()
    {
        return $this->morphMany(JournalEntry::class, 'type');
    }

    public function group()
    {
        return $this->belongsTo(Attribute::class);
    }

    public function account()
    {
        return AccountService::getAccountForModel(AccountHelper::ACCOUNT_SUPPLIERS,'purchases');
    }

    public function resolveResource()
    {
        return SupplierResource::class;
    }

    public function scopeActive(Builder $query): void
    {
        $query->where('status', UserStatusEnum::ACTIVE);
    }
    
    public function admins()
    {
        return $this->hasMany(SupplierAdmin::class);
    }
    
    public function paymentData()
    {
        return $this->hasOne(SuppliersPaymentData::class , 'supplier_id');
    }    

}
